import logging
from sqlalchemy.orm import Session, attributes
from sqlalchemy import func as sql_func

from watcher import enums
from watcher.db import (
    AbcMigration,
    Event,
    Notification,
    TaskMetric,
    Schedule,
    Shift,
    Unistat,
)
from watcher.db.base import dbconnect
from watcher.config import settings
from watcher.crud.shift import query_need_start_shifts
from watcher.logic.timezone import now
from watcher.tasks.base import lock_task

logger = logging.getLogger(__name__)


@lock_task(save_metrics=True, send_to_unistat=True)
@dbconnect
def calculate_unistat_metrics(session: Session):
    """
    Обновляем метрики
    """

    uncreated_shifts_subquery = (
        session.query(
            Schedule.id,
            # смотрим на смены с немного меньшим threshold
            (Schedule.threshold_day * settings.UNISTAT_SCHEDULE_THRESHOLD_MULTIPLIER).label('threshold'),
            sql_func.max(Shift.end).label('max_end'),
        )
        .join(Shift, Shift.schedule_id == Schedule.id)
        .filter(
            Schedule.state == enums.ScheduleState.active,
            Schedule.recalculate.is_(True)
        )
        .group_by(Schedule.id, Schedule.threshold_day * settings.UNISTAT_SCHEDULE_THRESHOLD_MULTIPLIER)
    ).subquery('uncreated_shifts_subquery')

    metrics = {
        'unprocessed_event_count': (
            session.query(Event).filter(Event.state.in_([
                enums.EventState.new,
                enums.EventState.scheduled,
            ]))
        ).count(),
        'failed_abc_migrations': (
            session.query(AbcMigration).filter(AbcMigration.status.in_([
                enums.AbcMigrationStatus.preparing_fail,
                enums.AbcMigrationStatus.finalizing_fail,
            ]))
        ).count(),
        'outdated_notifications': (
            session.query(Notification).filter(Notification.state == enums.NotificationState.outdated)
        ).count(),
        'unstarted_shifts': (
            query_need_start_shifts(db=session, future_time_shift=False)
        ).count(),
        'unfinished_shifts': (
            session.query(Shift).filter(
                Shift.status == enums.ShiftStatus.active,
                Shift.end < (now() - settings.SHIFT_FINISH_TIMEDELTA)
            )
        ).count(),
        'schedules_with_uncreated_shifts': (
            session.query(Schedule.id).filter(
                Schedule.id == uncreated_shifts_subquery.c.id,
                uncreated_shifts_subquery.c.max_end < (now() + uncreated_shifts_subquery.c.threshold),
            )
        ).count(),
        'unallocated_shifts': (
            session.query(Shift)
            .join(Schedule, Schedule.id == Shift.schedule_id)
            .filter(
                Schedule.state == enums.ScheduleState.active,
                Schedule.recalculate.is_(True),
                Shift.empty.is_(False),
                Shift.start > now(),
                Shift.staff_id.is_(None),
            )
        ).count(),
        'schedules_long_time_recalculation': (
            session.query(Schedule)
            .filter(
                Schedule.recalculation_in_process.is_(True),
                Schedule.updated_at < (now() + settings.UNISTAT_SCHEDULE_MAX_RECALCULATION_TIME)
            )
        ).count(),
    }

    # сколько секунд прошло с последнего успешного выполнения тасок
    for task_metric in session.query(TaskMetric).filter(TaskMetric.send_to_unistat.is_(True)).all():
        metrics[task_metric.task_name] = (now() - task_metric.last_success_end).total_seconds()

    for table_name, amount in (
        session.query(Event.table, sql_func.count())
        .filter(
            Event.state.in_([enums.EventState.new, enums.EventState.scheduled])
        ).group_by(Event.table)
    ).all():
        metrics[f'unprocessed_event_count_{table_name}'] = amount
    unistat_record = session.query(Unistat).first()
    if unistat_record:
        unistat_record.metrics = metrics
        attributes.flag_modified(unistat_record, 'metrics')  # У SQLAlchemy траблы с сохранением JSON
    else:
        new_record = Unistat(metrics=metrics)
        session.add(new_record)
