import pytest
import datetime
import responses
from unittest.mock import patch

from watcher.crud.schedule import get_schedule_by_id
from watcher.tasks.generating_shifts import initial_creation_of_shifts
from watcher.db import Schedule, ScheduleResponsible
from watcher import enums
from watcher.config import settings


def test_get_schedule(
    client, scope_session, service_factory, schedule_factory,
    assert_json_keys_value_equal, assert_count_queries
):
    schedule = schedule_factory()
    service = service_factory()
    schedule.show_in_services.append(service)
    scope_session.commit()
    with assert_count_queries(3):
        response = client.get(
            f'/api/watcher/v1/schedule/{schedule.id}',
        )
    assert response.status_code == 200, response.text
    data = response.json()
    expected = {
        'id': schedule.id,
        'name': schedule.name,
        'slug': schedule.slug,
        'service_id': schedule.service_id,
        'state': 'active',
        'recalculation_in_process': False,
        'is_important': False,
    }
    assert_json_keys_value_equal(data, expected)
    expected_service = {
        'id': schedule.service.id,
        'slug': schedule.service.slug,
        'name': schedule.service.name,
        'name_en': schedule.service.name_en,
    }
    assert_json_keys_value_equal(data['service'], expected_service)
    assert data['revision'] is None
    assert len(data['show_in_services']) == 1
    assert data['show_in_services'][0]['slug'] == service.slug
    assert data['schedules_group']['slug'] == schedule.schedules_group.slug
    assert data['schedules_group']['autogenerated'] == schedule.schedules_group.autogenerated


def test_schedule_with_revision(client, schedule_factory, revision_factory, assert_count_queries):
    schedule = schedule_factory()
    revision_factory(schedule=schedule, state=enums.RevisionState.disabled)
    next_revision = revision_factory(schedule=schedule)
    revision = revision_factory(schedule=schedule, next=next_revision)
    with assert_count_queries(4):
        response = client.get(
            f'/api/watcher/v1/schedule/{schedule.id}',
            params={'fields': 'revision'},
        )
    assert response.status_code == 200, response.text
    data = response.json()
    assert data['id'] == schedule.id
    assert data['revision']['id'] == revision.id
    assert data['revision']['apply_datetime'] == revision.apply_datetime.isoformat()
    assert data['show_in_services'] == []


def test_list_schedule(client, schedule_factory, revision_factory, scope_session,):
    expected = [schedule_factory() for _ in range(2)]
    expected[0].is_important = True
    scope_session.commit()
    revision_factory(schedule=expected[0])
    response = client.get(
        '/api/watcher/v1/schedule/',
    )
    assert response.status_code == 200, response.text
    data = response.json()['result']
    assert len(data) == 2
    assert {obj['id'] for obj in data} == {obj.id for obj in expected}
    assert data[0]['revision'] is None
    assert data[0]['is_important'] is not data[1]['is_important']


@pytest.mark.parametrize('link', (None, 'service', 'show_in_services'))
def test_filter_schedule(client, scope_session, service_factory,  schedule_factory, revision_factory, link):
    """
    проверяем работу запроса, который использует фронтенд
    для фильтрации дежурств в сервисе
    """
    service = service_factory()
    schedule = schedule_factory(name='поддержка')
    if link == 'service':
        schedule.service = service
        scope_session.commit()
    elif link == 'show_in_services':
        schedule.show_in_services.append(service)
        scope_session.commit()
    response = client.get(
        '/api/watcher/v1/schedule/',
        params={'filter': f'service_id={service.id}|show_in_services.id={service.id},'
                          'name__ilike=поддер|slug__ilike=поддер'}
    )
    assert response.status_code == 200, response.text
    data = response.json()['result']
    if not link:
        assert len(data) == 0
    else:
        assert len(data) == 1
        assert data[0]['id'] == schedule.id


@pytest.mark.parametrize('has_revision', (True, False))
def test_list_schedule_with_revision(client, schedule_factory, revision_factory, has_revision):
    schedule = schedule_factory()
    revision_factory(schedule=schedule, state=enums.RevisionState.disabled)
    next_revision = revision_factory(schedule=schedule)
    revision = revision_factory(schedule=schedule, next=next_revision)
    schedule_factory()

    params = {'fields': 'revision'}
    if has_revision:
        params['has_revision'] = True
    response = client.get(
        '/api/watcher/v1/schedule/',
        params=params,
    )
    assert response.status_code == 200, response.text
    data = response.json()['result']

    with_revision = data[0]
    if has_revision:
        assert len(data) == 1
    else:
        assert len(data) == 2
        assert data[0]['revision'] is None
        with_revision = data[1]

    assert with_revision['id'] == schedule.id
    assert with_revision['revision']['id'] == revision.id
    assert with_revision['revision']['apply_datetime'] == revision.apply_datetime.isoformat()


@pytest.mark.parametrize(
    'services_filters',
    (
        'my_schedules',
        'other_schedules',
        'my_and_other_schedules',
        'complex_filter',
        'all',
    )
)
def test_list_schedule_show_in_services(client, schedule_factory, revision_factory, services_filters):
    schedule_1 = schedule_factory()
    schedule_2 = schedule_factory()
    schedule_3 = schedule_factory(show_in_services=[schedule_1.service])
    schedule_4 = schedule_factory(show_in_services=[schedule_1.service, schedule_2.service])

    if services_filters == 'my_schedules':
        params = {'filter': f'service_id={schedule_1.service_id}'}
    elif services_filters == 'other_schedules':
        params = {'filter': f'show_in_services.id={schedule_1.service_id}'}
    elif services_filters == 'my_and_other_schedules':
        params = {'filter': f'service_id={schedule_1.service_id}|show_in_services.id={schedule_1.service_id}'}
    elif services_filters == 'complex_filter':
        params = {
            'filter': ','.join([
                f'service_id={schedule_1.service_id}|show_in_services.id={schedule_1.service_id}',
                f'service_id={schedule_3.service_id}',
            ])
        }
    else:
        params = None

    response = client.get(
        '/api/watcher/v1/schedule/',
        params=params,
    )
    assert response.status_code == 200, response.text
    data = response.json()['result']

    if services_filters == 'my_schedules':
        assert len(data) == 1
        assert {service['id'] for service in data} == {schedule_1.id, }
    elif services_filters == 'other_schedules':
        assert len(data) == 2
        assert {service['id'] for service in data} == {schedule_3.id, schedule_4.id, }
    elif services_filters == 'my_and_other_schedules':
        assert len(data) == 3
        assert {service['id'] for service in data} == {schedule_1.id, schedule_3.id, schedule_4.id, }
    elif services_filters == 'complex_filter':
        assert len(data) == 1
        assert {service['id'] for service in data} == {schedule_3.id, }
    else:
        assert len(data) == 4


def test_delete_schedule(client, schedule_factory, scope_session):
    schedule = schedule_factory()
    response = client.delete(
        f'/api/watcher/v1/schedule/{schedule.id}',
    )
    assert response.status_code == 204, response.text
    assert not scope_session.query(Schedule).filter(
        Schedule.id == schedule.id,
    ).count()


@responses.activate
@pytest.mark.parametrize('is_notification_enabled', (False, True))
@pytest.mark.parametrize('with_services', (False, True))
def test_create_schedule(
    client, service_factory, schedules_group_factory,
    is_notification_enabled, scope_session, test_request_user,
    with_services
):
    group = schedules_group_factory()
    service = service_factory()
    initial_data = {
        'slug': 'test_schedule_slug',
        'name': 'test schedule name',
        'service_id': service.id,
        'schedules_group_id': group.id,
        'description': 'test schedule description',
        'days_before_vacation': 86400,
    }
    if not is_notification_enabled:
        initial_data['days_for_notify_of_problems'] = None
        initial_data['days_for_notify_of_begin'] = None
    if with_services:
        linked_service = service_factory()
        initial_data['show_in_services'] = [linked_service.id]
    responses.add(
        responses.GET,
        'https://abc-back.test.yandex-team.ru/api/v4/duty/schedules/',
        status=200,
        json={'results': []}
    )
    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 201, response.text
    data = response.json()
    expected = {
        'name': 'test schedule name',
        'slug': 'test_schedule_slug',
        'service_id': service.id,
        'description': 'test schedule description',
        'state': 'active',
        'recalculate': True,
        'days_for_notify_of_problems': 1209600 if is_notification_enabled else None,
        'days_for_notify_of_begin': [0, 86400, 86400 * 7] if is_notification_enabled else None,
        'days_before_vacation': 86400,
        'days_after_vacation': 0,
    }
    for key, value in expected.items():
        assert data[key] == value
    assert 'id' in data
    if with_services:
        assert data['show_in_services'][0]['slug'] == linked_service.slug
    else:
        assert data['show_in_services'] == []
    assert (scope_session.query(ScheduleResponsible)
            .filter(ScheduleResponsible.responsible_id == test_request_user.id)
            .count()
            )
    responsible = scope_session.query(ScheduleResponsible).first()
    assert responsible.responsible_id == test_request_user.id
    assert responsible.schedule_id == data['id']


@responses.activate
def test_create_schedule_wrong_data(client, schedules_group_factory, scope_session):
    group = schedules_group_factory()
    responses.add(
        responses.GET,
        'https://abc-back.test.yandex-team.ru/api/v4/duty/schedules/',
        status=200,
        json={'results': []}
    )
    initial_data = {
        'slug': 'test_schedule_slug',
        'name': 'test schedule name',
        'service_id': 100500,
        'schedules_group_id': group.id,
    }
    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    # такого сервиса нет
    assert response.status_code == 404, response.text


@responses.activate
@pytest.mark.parametrize('same_service_watcher', [True, False])
@pytest.mark.parametrize('same_service_abc', [True, False])
def test_create_schedule_same_slug_watcher(
    client, schedules_group_factory, service_factory,
    schedule_factory, scope_session, same_service_watcher,
    same_service_abc,
):
    slug = 'schedule_slug'
    group = schedules_group_factory()
    service = service_factory()
    service_1 = service_factory()
    initial_data = {
        'slug': slug,
        'name': 'test schedule name',
        'service_id': service.id,
        'schedules_group_id': group.id,
    }
    schedule_factory(slug=slug, service=service if same_service_watcher else service_1)
    responses.add(
        responses.GET,
        'https://abc-back.test.yandex-team.ru/api/v4/duty/schedules/',
        status=200,
        json={'results': [{'slug': slug}] if same_service_abc else []}
    )

    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    expected_code = 201
    if same_service_abc or same_service_watcher:
        expected_code = 400
    assert response.status_code == expected_code, response.text


def test_patch_schedule(client, service_factory, schedule_factory, scope_session):
    schedule = schedule_factory()
    linked_service = service_factory()
    new_linked_service = service_factory()
    schedule.show_in_services.append(linked_service)
    scope_session.commit()

    patch_data = {
        'name': 'test schedule name',
        'show_in_services': [new_linked_service.id],
        'days_for_notify_of_begin': [0, 86400],
        'days_for_notify_of_problems': None,
    }
    response = client.patch(
        f'/api/watcher/v1/schedule/{schedule.id}',
        json=patch_data,
    )
    assert response.status_code == 200, response.text
    data = response.json()
    assert data['name'] == 'test schedule name'

    scope_session.refresh(schedule)
    assert schedule.name == 'test schedule name'
    assert len(schedule.show_in_services) == 1
    assert schedule.show_in_services[0] == new_linked_service
    assert schedule.days_for_notify_of_begin == [
        datetime.timedelta(days=0),
        datetime.timedelta(days=1),
    ]
    assert schedule.days_for_notify_of_problems is None


def test_patch_schedule_group_id(client, service_factory, schedule_factory, scope_session):
    schedule = schedule_factory()
    current_group_id = schedule.schedules_group_id

    patch_data = {
        'schedules_group_id': None,
    }
    response = client.patch(
        f'/api/watcher/v1/schedule/{schedule.id}',
        json=patch_data,
    )
    assert response.status_code == 200, response.text
    data = response.json()
    assert data['schedules_group']['id'] != current_group_id

    scope_session.refresh(schedule)
    assert schedule.schedules_group_id != current_group_id
    assert schedule.schedules_group.autogenerated


@pytest.mark.parametrize(
    ('initial', 'target'),
    [
        (enums.ScheduleState.active, enums.ScheduleState.disabled),
        (enums.ScheduleState.active, enums.ScheduleState.disabled),
        (enums.ScheduleState.active, enums.ScheduleState.active),
        (enums.ScheduleState.disabled, enums.ScheduleState.disabled),
    ]
)
def test_patch_schedule_state(scope_session, client, schedule_factory, initial, target):
    schedule = schedule_factory(state=initial)
    patch_data = {
        'state': target,
    }
    with patch('watcher.api.routes.schedule.delete_disabled_shifts') as mock_disable:
        with patch('watcher.api.routes.schedule.proceed_new_shifts') as mock_enable:
            response = client.patch(
                f'/api/watcher/v1/schedule/{schedule.id}',
                json=patch_data,
            )
    assert response.status_code == 200, response.text
    scope_session.refresh(schedule)
    assert schedule.state == target

    if initial == target:
        mock_disable.delay.assert_not_called()
        mock_enable.delay.assert_not_called()
    elif target == enums.ScheduleState.disabled:
        mock_disable.delay.assert_called_once()
        mock_enable.delay.assert_not_called()
    else:
        mock_disable.delay.assert_not_called()
        mock_enable.delay.assert_called_once()


def test_patch_schedule_wrong_state(client, schedule_factory):
    schedule = schedule_factory()
    patch_data = {
        'state': 'smth',
    }
    response = client.patch(
        f'/api/watcher/v1/schedule/{schedule.id}',
        json=patch_data,
    )
    assert response.status_code == 422, response.text
    data = response.json()
    assert data['detail'][0]['msg']['ru'] == (
        'value is not a valid enumeration '
        'member; permitted: \'active\', \'disabled\''
    )


def test_patch_schedule_invalid_group_id(client, schedule_factory):
    schedule = schedule_factory()
    patch_data = {
        'schedules_group_id': 100500,
    }
    response = client.patch(
        f'/api/watcher/v1/schedule/{schedule.id}',
        json=patch_data,
    )
    assert response.status_code == 400, response.text
    data = response.json()
    assert data['context']['error'] == (
        'IntegrityError(\'(psycopg2.errors.ForeignKeyViolation) '
        'insert or update on table "schedule" violates foreign '
        'key constraint "schedule_schedules_group_id_fkey"\\n'
        'DETAIL:  Key (schedules_group_id)=(100500) is not '
        'present in table "schedules_group".\\n\')'
    )


def test_patch_schedule_not_existed(client):
    patch_data = {
        'name': 'test',
    }
    response = client.patch(
        '/api/watcher/v1/schedule/100500',
        json=patch_data,
    )
    assert response.status_code == 404, response.text


@pytest.mark.parametrize('valid_data', (False, True))
@pytest.mark.parametrize('rotation', (None, 'backup_is_next_primary'))
def test_patch_schedule_intervalrotation(client, schedule_factory, interval_factory, slot_factory, composition_factory,
                                         valid_data, rotation, assert_count_queries):
    schedule = schedule_factory()
    composition = composition_factory()
    interval_1 = interval_factory(schedule=schedule)
    interval_2 = interval_factory(schedule=schedule)
    if valid_data:
        slot_factory(interval=interval_1, composition=composition, is_primary=False)
        slot_factory(interval=interval_1, composition=composition, is_primary=True)
        slot_factory(interval=interval_2, composition=composition, is_primary=False)
        slot_factory(interval=interval_2, composition=composition, is_primary=True)
        queries_count = 5
    else:
        slot_factory(interval=interval_1, composition=composition, is_primary=False)
        slot_factory(interval=interval_2, composition=composition, is_primary=True)
        queries_count = 2

    patch_data = {'name': 'new schedule name'}
    if rotation:
        patch_data['rotation'] = rotation
    else:
        queries_count = 4

    schedule_id = schedule.id
    with assert_count_queries(queries_count):
        with patch('watcher.api.routes.schedule.start_people_allocation.delay') as mock_start_people_allocation:
            response = client.patch(
                f'/api/watcher/v1/schedule/{schedule_id}',
                json=patch_data,
            )
    if not rotation:
        assert response.status_code == 200, response.text
    elif valid_data:
        assert response.status_code == 200, response.text
        mock_start_people_allocation.assert_called_once()
    else:
        assert response.status_code == 400, response.text
        data = response.json()
        assert data['detail'][0]['msg'] == {
            'ru': 'В переданных сменах должен быть один пул дежурных, а также по одному запасному и основному дежурному.',
            'en': 'In the uploaded shifts there must be one same duty pool, as well as one backup and primary duty person in each.',
        }


@responses.activate
def test_create_schedule_without_group(client, service_factory, scope_session):
    service = service_factory()
    initial_data = {
        'slug': 'test_schedule_slug',
        'name': 'test schedule name',
        'service_id': service.id,
    }
    responses.add(
        responses.GET,
        'https://abc-back.test.yandex-team.ru/api/v4/duty/schedules/',
        status=200,
        json={'results': []}
    )
    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 201, response.text
    schedule = get_schedule_by_id(scope_session, response.json()['id'])
    assert schedule.schedules_group


def test_bind_to_autogenerated(client, service_factory, schedule_factory, schedules_group_factory, scope_session):
    group = schedules_group_factory(autogenerated=True)
    service = service_factory()
    initial_data = {
        'slug': 'test_schedule_slug',
        'name': 'test schedule name',
        'service_id': service.id,
        'schedules_group_id': group.id
    }
    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 400, response.text
    assert response.json()['error'] == 'schedule_group_is_autogenerated'

    schedule = schedule_factory()

    patch_data = {
        'name': 'test schedule name',
        'schedules_group_id': group.id
    }
    response = client.patch(
        f'/api/watcher/v1/schedule/{schedule.id}',
        json=patch_data,
    )
    assert response.status_code == 400, response.text
    assert response.json()['error'] == 'schedule_group_is_autogenerated'


def test_custom_exception_format(client, scope_session):
    initial_data = {
        'slug': 'test_schedule_slug',
        'name': 'test schedule name',
        'service_id': 'test',
    }

    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 422, response.text
    assert response.json() == {
        "detail": [
            {
                "loc": ["body", "service_id"],
                "msg": {
                    "ru": "value is not a valid integer",
                    "en": "value is not a valid integer"
                },
                "type": "type_error.integer"}
        ]
    }


def test_patch_schedule_not_existed_field(schedule_factory, client):
    schedule = schedule_factory()
    patch_data = {
        'name': 'test',
        'some_unknown_field': '234234',
    }
    response = client.patch(
        f'/api/watcher/v1/schedule/{schedule.id}',
        json=patch_data,
    )
    assert response.status_code == 422, response.text
    assert response.json()['detail'][0]['msg'] == {
        'ru': 'Unknown fields: {\'some_unknown_field\'}',
        'en': 'Unknown fields: {\'some_unknown_field\'}',
    }


@pytest.mark.parametrize(
    'data,expected_error',
    [
        [
            {'days_for_notify_of_begin': [86400 * 4, 86400 * 50]},
            f'Максимально допустимое значение - {settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS} дней'
        ],
        [
            {'days_for_notify_of_begin': [86400 * 50, 86400 * 4]},
            f'Максимально допустимое значение - {settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS} дней'
        ],
        [
            {'days_for_notify_of_begin': [86400 * 31]},
            f'Максимально допустимое значение - {settings.MAX_PROBLEM_OR_NOTIFICATION_DAYS} дней'
        ],
        [
            {'threshold_day': 259200 * 61},
            f'Максимально допустимое значение - {settings.MAX_THRESHOLD_DAYS} дней'
        ],
        [
            {'threshold_day': 259200 * 5},
            f'Минимально допустимое значение - {settings.MIN_THRESHOLD_DAYS} дней'
        ],
        [
            {'days_before_vacation': 86400 * 31},
            f'Максимально допустимое значение - {settings.MAX_DAYS_BEFORE_OR_AFTER_VACATION} дней'
        ],
        [
            {'days_after_vacation': 86400 * 31},
            f'Максимально допустимое значение - {settings.MAX_DAYS_BEFORE_OR_AFTER_VACATION} дней'
        ],
    ]
)
def test_validate_days_fields(client, scope_session, data, expected_error):
    initial_data = {
        'slug': 'test_schedule_slug',
        'name': 'test schedule name',
        'service_id': 123,
    }

    initial_data.update(data)

    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 400, response.text
    assert response.json()['detail'][0]['msg']['ru'] == expected_error


def test_inactive_service(client, scope_session, service_factory):
    service = service_factory(state='deleted')
    initial_data = {
        'slug': 'test_schedule_slug',
        'name': 'test schedule name',
        'service_id': service.id,
    }

    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 400, response.text
    assert response.json()['detail'][0]['msg']['ru'] == 'Сервис должен находиться в активном статусе'


@pytest.mark.parametrize('slug_error_type', ['length_min', 'length_max', 'value'])
def test_create_schedule_with_invalid_slug(client, slug_error_type):
    if slug_error_type == 'length_min':
        slug = 'qq'
    elif slug_error_type == 'length_max':
        slug = 'q' * 51
    else:
        slug = 'invalid slug'
    initial_data = {
        'slug': slug,
        'name': 'valid schedule name',
    }
    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 400, response.text
    massage = response.json()['detail'][0]['msg']
    if slug_error_type == 'value':
        assert massage['ru'] == 'Slug должен состоять из латинских букв, цифр, знаков подчеркивания или дефиса.'
        assert massage['en'] == 'Slug should contain only latin, numeric, underscore and hyphens characters.'
    else:
        assert massage['ru'] == 'Минимальная длина слага - 3 символа, максимальная - 50. Поле обязательно.'
        assert massage['en'] == 'The minimum length of the slug is 3 character, the maximum is 50. ' \
                                'The field is required.'


@pytest.mark.parametrize('name_error_type', ['length_min', 'length_max'])
def test_create_schedule_with_invalid_name(client, name_error_type):
    if name_error_type == 'length_min':
        name = ''
    else:
        name = 'q' * 151
    initial_data = {
        'slug': 'valid_schedule_slug',
        'name': name,
    }
    response = client.post(
        '/api/watcher/v1/schedule/',
        json=initial_data,
    )
    assert response.status_code == 400, response.text
    massage = response.json()['detail'][0]['msg']

    assert massage['ru'] == 'Минимальная длина названия - 1 символ, максимальная - 150. Поле обязательно.'
    assert massage['en'] == 'The minimum length of the name is 1 character, the maximum is 150.' \
                            ' The field is required.'


def test_list_schedule_responsibles(
    client, schedule_factory, staff_factory,
    schedule_responsible_factory, permission_factory,
    user_with_permissions, service_factory, member_factory, role_factory
):

    schedule = schedule_factory()
    schedule_responsible_factory(schedule=schedule, responsible=user_with_permissions)

    schedule1 = schedule_factory()
    staff1 = staff_factory()
    schedule_responsible_factory(schedule=schedule1, responsible=staff1)

    schedule2 = schedule_factory()
    ancestor = service_factory()
    schedule2.service.ancestors = [{'id': ancestor.id}]
    member_factory(
        staff=user_with_permissions,
        service=ancestor,
        role=role_factory(code=settings.RESPONSIBLE_ROLE_CODE)
    )

    schedule3 = schedule_factory()
    ancestor1 = service_factory()
    schedule3.service.ancestors = [{'id': ancestor1.id}]
    member_factory(staff=user_with_permissions, service=ancestor1)

    response = client.get(
        '/api/watcher/v1/schedule/',
        params={'fields': 'is_responsible'},
    )
    assert response.status_code == 200, response.text
    data = response.json()['result']
    assert len(data) == 4
    expected = {
        schedule.id: True,
        schedule1.id: False,
        schedule2.id: True,
        schedule3.id: False
    }

    data = {
        obj['id']: obj['is_responsible'] for obj in data
    }
    assert data == expected


@pytest.mark.parametrize('get_fields', (False, True))
def test_recalculate_schedule(client, schedule_factory, interval_factory, revision_factory, slot_factory,
                              assert_count_queries, get_fields):
    """
    Проверяем, что start_people_allocation запускается,
    Проверяем, что возвращается is_responsible и revision если указано в fields
    """
    schedule = schedule_factory()
    revision = revision_factory(schedule=schedule)
    interval = interval_factory(schedule=schedule, revision=revision)
    slot_factory(interval=interval)

    with patch('watcher.api.routes.schedule.start_people_allocation.delay') as mock_start_people_allocation:
        url = f'/api/watcher/v1/schedule/{schedule.id}/recalculate'
        params = None
        query_count = 3
        if get_fields:
            params = {'fields': 'is_responsible,revision'}
            query_count += 1
        with assert_count_queries(query_count):
            # SELECT intranet_staff
            # SELECT schedule joined
            # (if get_fields) SELECT revision
            # UPDATE schedule
            response = client.post(
                url=url,
                params=params
            )
    assert response.status_code == 200, response.text
    data = response.json()
    assert data['id'] == schedule.id
    if get_fields:
        assert data['is_responsible']
        assert data['revision']['id'] == revision.id

    mock_start_people_allocation.assert_called_once()
    assert mock_start_people_allocation.call_args.kwargs.get('schedules_group_id') == schedule.schedules_group_id
    assert mock_start_people_allocation.call_args.kwargs.get('force_task_delay')


@pytest.mark.parametrize('show_in_staff', [True, False])
def test_show_in_staff(
    client, schedule_factory, assert_count_queries,
    slot_factory, interval_factory, scope_session, show_in_staff,
):
    schedule = schedule_factory()

    interval_prev = interval_factory(schedule=schedule)
    interval = interval_factory(schedule=schedule)
    interval_prev.revision.next = interval.revision
    scope_session.commit()

    slot_factory(interval=interval_prev, show_in_staff=not show_in_staff)
    slot_factory(interval=interval, show_in_staff=show_in_staff)

    with assert_count_queries(3):
        response = client.get(
            '/api/watcher/frontend/schedule/',
            params={'fields': 'show_in_staff'},
        )
    assert response.status_code == 200, response.text
    data = response.json()['result']
    assert len(data) == 1
    assert data[0]['show_in_staff'] is show_in_staff


def test_schedule_problems(
    client, schedule_factory, shift_factory, problem_factory,
    slot_factory, interval_factory, assert_count_queries,
    gap_factory,
):
    schedule_1 = schedule_factory()
    schedule_2 = schedule_factory()

    slot_1 = slot_factory(interval=interval_factory(schedule=schedule_1), show_in_staff=True)
    slot_2 = slot_factory(interval=interval_factory(schedule=schedule_2))

    shift_11 = shift_factory(schedule=schedule_1, slot=slot_1)
    shift_12 = shift_factory(schedule=schedule_1, slot=slot_1)
    shift_21 = shift_factory(schedule=schedule_2, slot=slot_2)
    shift_22 = shift_factory(schedule=schedule_2, slot=slot_2)

    problem_factory(shift=shift_11, status=enums.ProblemStatus.new)
    problem_factory(shift=shift_11, status=enums.ProblemStatus.reported)
    problem_factory(shift=shift_12, status=enums.ProblemStatus.reported)
    problem_factory(shift=shift_12, status=enums.ProblemStatus.resolved)

    problem_factory(
        shift=shift_21,
        status=enums.ProblemStatus.new,
        duty_gap=gap_factory(),
    )
    problem_factory(shift=shift_21, status=enums.ProblemStatus.reported)
    problem_factory(shift=shift_22, status=enums.ProblemStatus.reported)
    problem_factory(shift=shift_22, status=enums.ProblemStatus.resolved)

    with assert_count_queries(2):
        response = client.get(
            '/api/watcher/frontend/schedule/',
        )
    assert response.status_code == 200, response.text
    data = response.json()['result']

    assert len(data) == 2
    for schedule in data:
        actual_problems = schedule['actual_problems']
        assert len(actual_problems) == 3
        assert set(problem['status'] for problem in actual_problems) == {'new', 'reported'}


def test_gap_settings_deactivation_on_schedule_deletion(
    client, schedule_factory, interval_factory, revision_factory, slot_factory, manual_gap_settings_factory,
    scope_session
):
    schedule = schedule_factory()
    interval = interval_factory(schedule=schedule, revision=revision_factory(schedule=schedule))
    slot_factory(interval=interval)

    gap_settings = manual_gap_settings_factory(is_active=True)
    gap_settings.schedules = [schedule]  # при удалении расписания должен деактивироваться
    scope_session.commit()

    initial_creation_of_shifts(schedule.id)

    response = client.delete(
        f'/api/watcher/v1/schedule/{schedule.id}',
    )
    assert response.status_code == 204, response.text
    assert not scope_session.query(Schedule).filter(
        Schedule.id == schedule.id,
    ).count()

    scope_session.refresh(gap_settings)
    assert gap_settings.is_active is False
