from fastapi import status
import pytest
import datetime

from watcher.db import SchedulesGroup, ScheduleGroupResponsible


def test_get_schedule_group(client, schedules_group_factory, assert_json_keys_value_equal, staff_factory,
                            group_responsible_factory, scope_session):
    schedule_group = schedules_group_factory()
    staff = staff_factory()
    staff1 = staff_factory()

    group_responsible_factory(
        schedule_group=schedule_group,
        responsible=staff
    )

    group_responsible_factory(
        schedule_group=schedule_group,
        responsible=staff1
    )
    response = client.get(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}'
    )

    assert response.status_code == 200, response.text
    data = response.json()
    assert data['id'] == schedule_group.id
    assert data['name'] == schedule_group.name
    assert data['slug'] == schedule_group.slug
    assert data['timeout_between_shifts'] == schedule_group.timeout_between_shifts.total_seconds()
    assert len(data['responsibles']) == 2
    assert {responsible['uid'] for responsible in data['responsibles']} == {staff.uid, staff1.uid}


def test_list_schedule_group(client, schedules_group_factory, staff_factory, group_responsible_factory):
    schedules_group_factory(autogenerated=True)
    expected = [schedules_group_factory() for _ in range(2)]
    staff = staff_factory()

    group_responsible_factory(
        schedule_group=expected[0],
        responsible=staff
    )
    response = client.get(
        '/api/watcher/v1/schedule_group/',
        params={'filter': f'id={expected[0].id},id={expected[1].id}'}
    )
    assert response.status_code == status.HTTP_200_OK
    assert response.text
    data = response.json()['result']
    assert len(data) == 2
    assert {obj['id'] for obj in data} == {obj.id for obj in expected}
    assert data[1]['id'] == expected[0].id
    assert data[1]['responsibles'][0]['uid'] == staff.uid


def test_delete_schedule_group(client, schedules_group_factory, scope_session):
    schedule_group = schedules_group_factory()
    response = client.delete(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}'
    )
    assert response.status_code == status.HTTP_204_NO_CONTENT
    assert not response.text
    assert not scope_session.query(SchedulesGroup).filter(
        SchedulesGroup.id == schedule_group.id,
    ).count()


def test_create_schedule_group(client, scope_session, test_request_user, assert_count_queries):
    initial_data = {
        'name': 'имя группы',
        'slug': 'test_schedule_slug',
        'timeout_between_shifts': str(datetime.timedelta(hours=4))
    }
    with assert_count_queries(7):
        # staff
        # check unique slug
        # insert data
        # insert responsible
        # select group
        # select responsibles
        # select schedules
        response = client.post(
            '/api/watcher/v1/schedule_group/',
            json=initial_data
        )
    assert response.status_code == status.HTTP_201_CREATED, response.text
    assert response.text
    data = response.json()
    assert data['name'] == initial_data['name']
    assert data['slug'] == initial_data['slug']
    assert data['timeout_between_shifts'] == 3600 * 4.0
    assert len(data['responsibles']) == 1
    assert data['responsibles'][0]['uid'] == test_request_user.uid

    responsible = scope_session.query(ScheduleGroupResponsible).first()
    assert responsible.responsible_id == test_request_user.id
    assert responsible.schedule_group_id == data['id']


@pytest.mark.parametrize('method', ('post', 'patch'))
def test_unique_schedule_group_slug(client, schedules_group_factory, method):
    schedule_group = schedules_group_factory(slug='test_schedule_slug')
    if method == 'post':
        data = {
            'name': 'имя группы',
            'slug': 'test_schedule_slug',
        }
        url = '/api/watcher/v1/schedule_group/'
    else:
        schedules_group_factory(slug='test_schedule_slug1')
        data = {
            'slug': 'test_schedule_slug1',
        }
        url = f'/api/watcher/v1/schedule_group/{schedule_group.id}'
    response = getattr(client, method)(url, json=data)
    assert response.status_code == status.HTTP_400_BAD_REQUEST
    assert response.json()['error'] == 'schedule_group_slug_must_be_unique'


def test_patch_schedule_group(client, schedules_group_factory, schedule_factory, scope_session):
    schedule_group = schedules_group_factory()
    schedule_factory(schedules_group=schedule_group)
    patch_data = {
        'name': 'test schedule group name',
    }
    response = client.patch(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}',
        params={'fields': 'schedules.is_responsible'},
        json=patch_data,
    )
    assert response.status_code == status.HTTP_200_OK
    assert response.text
    data = response.json()
    assert data['name'] == patch_data['name']
    assert data['schedules'][0]['is_responsible'] is True

    scope_session.refresh(schedule_group)
    assert schedule_group.name == patch_data['name']


def test_patch_schedule_not_existed(client):
    patch_data = {
        'name': 'test',
    }
    response = client.patch(
        '/api/watcher/v1/schedule_group/100500',
        json=patch_data
    )
    assert response.status_code == status.HTTP_404_NOT_FOUND, response.text


def test_delete_schedule_group_with_schedules(client, schedule_factory, schedules_group_factory, scope_session):
    schedule_group = schedules_group_factory()
    schedule_factory(
        schedules_group_id=schedule_group.id,
        schedules_group=schedule_group,
    )
    response = client.delete(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}'
    )
    assert response.status_code == status.HTTP_400_BAD_REQUEST
    assert response.json()['error'] == 'schedule_group_has_reference'


@pytest.mark.parametrize('group_responsible', (True, False))
@pytest.mark.parametrize('is_responsible_param', (True, False, None))
def test_group_responsible(
    client, group_responsible_factory,
    schedules_group_factory, scope_session, staff_factory,
    test_request_user, group_responsible, is_responsible_param,
):
    schedule_group = schedules_group_factory()
    if group_responsible:
        group_responsible_factory(
            schedule_group=schedule_group,
            responsible=test_request_user,
        )

    if is_responsible_param is not None:
        response = client.get(
            '/api/watcher/v1/schedule_group/',
            params={'is_responsible': is_responsible_param},
        )
    else:
        response = client.get('/api/watcher/v1/schedule_group/')

    assert response.status_code == 200, response.text
    data = response.json()['result']
    if is_responsible_param and not group_responsible:
        assert len(data) == 0
    else:
        assert len(data) == 1


def test_put_schedule_group_responsibles(
    client, schedules_group_factory, group_responsible_factory,
    staff_factory, permission_factory, scope_session
):
    schedule_group = schedules_group_factory()
    staff = staff_factory()
    staff1 = staff_factory()

    group_responsible_factory(
        schedule_group=schedule_group,
        responsible=staff
    )

    group_responsible_factory(
        schedule_group=schedule_group,
        responsible=staff1
    )

    current_values = {ref.responsible_id for ref in scope_session.query(ScheduleGroupResponsible).filter(
        ScheduleGroupResponsible.schedule_group_id == schedule_group.id
    ).all()}

    assert current_values == {staff.id, staff1.id}

    patch_data = {'staff': [staff.id]}
    response = client.put(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}',
        json=patch_data,
    )
    data = response.json()
    assert data['id'] == schedule_group.id
    assert data['name'] == schedule_group.name
    assert data['slug'] == schedule_group.slug
    assert len(data['responsibles']) == 1
    assert data['responsibles'][0]['uid'] == int(staff.uid)
    assert response.status_code == 200
    current_values = {ref.responsible_id for ref in scope_session.query(ScheduleGroupResponsible).filter(
        ScheduleGroupResponsible.schedule_group_id == schedule_group.id
    ).all()}
    assert current_values == {staff.id}

    patch_data = {'staff': [staff.id, staff1.id]}
    response = client.put(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}',
        json=patch_data,
    )

    current_values = {ref.responsible_id for ref in scope_session.query(ScheduleGroupResponsible).filter(
        ScheduleGroupResponsible.schedule_group_id == schedule_group.id
    ).all()}

    data = response.json()
    assert data['id'] == schedule_group.id
    assert data['name'] == schedule_group.name
    assert data['slug'] == schedule_group.slug
    assert len(data['responsibles']) == 2
    assert data['responsibles'][0]['uid'] == int(staff.uid)
    assert data['responsibles'][1]['uid'] == int(staff1.uid)

    assert response.status_code == 200
    assert current_values == {staff.id, staff1.id}


def test_put_schedule_group_schedules(scope_session, client, schedules_group_factory, schedule_factory):
    schedule_group = schedules_group_factory()
    schedule_1 = schedule_factory(schedules_group=schedule_group)
    schedule_2 = schedule_factory(schedules_group=schedule_group)
    schedule_3 = schedule_factory()

    scope_session.refresh(schedule_group)
    assert {schedule.id for schedule in schedule_group.schedules} == {schedule_1.id, schedule_2.id}

    patch_data = {'schedule_ids': [schedule_2.id, schedule_3.id]}
    response = client.put(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}/schedules',
        json=patch_data,
    )

    assert response.status_code == 200
    data = response.json()

    assert {schedule['id'] for schedule in data['schedules']} == {schedule_2.id, schedule_3.id}


def test_put_schedule_group_responsibles_autogenerate(client, staff_factory, schedules_group_factory, scope_session,
                                                      group_responsible_factory):
    schedule_group = schedules_group_factory(autogenerated=True)
    staff = staff_factory()

    group_responsible_factory(
        schedule_group=schedule_group,
        responsible=staff
    )
    patch_data = {'staff': [staff.id]}
    response = client.put(
        f'/api/watcher/v1/schedule_group/{schedule_group.id}',
        json=patch_data,
    )
    assert response.status_code == 400, response.text
    assert response.json()['error'] == 'schedule_group_is_autogenerated'


@pytest.mark.parametrize('list_route', (True, False))
def test_is_responsible_for_schedules(
    client, schedule_factory, schedule_responsible_factory,
    user_with_permissions, list_route,
):
    schedule = schedule_factory()
    schedule_responsible_factory(schedule=schedule, responsible=user_with_permissions)
    schedule1 = schedule_factory(
        schedules_group=schedule.schedules_group
    )
    if list_route:
        url = '/api/watcher/v1/schedule_group/'
    else:
        url = f'/api/watcher/v1/schedule_group/{schedule.schedules_group_id}'
    response = client.get(url, params={'fields': 'schedules.is_responsible'})
    assert response.status_code == 200, response.text
    data = response.json()
    if list_route:
        schedules = data['result'][0]['schedules']
    else:
        schedules = data['schedules']
    assert len(schedules) == 2
    assert {
        obj['id']: obj['is_responsible']
        for obj in schedules
    } == {schedule.id: True, schedule1.id: False}
