# encoding: utf-8
from __future__ import unicode_literals

import itertools
import json
import logging
import time

from tornado import gen
from ylog.context import put_to_context

from intranet.webauth.lib import role_cache, step as webauth_step
from intranet.webauth.lib.utils import get_user_ip

logger = logging.getLogger(__name__)

DEFAULT_STEPS_PIPELINE = ['cert', 'token', 'cookies']


class Authorizer(object):
    @staticmethod
    def from_request(request, forced_domain=None, scopes_to_check=[], simulated=False):
        args = request.query_arguments
        # every argument value is a list of 1+ elements. We accept only single-valued arguments
        for key in args:
            value = args[key]
            if type(value) == list and len(value) > 1:
                return
            args[key] = value[0]
        return Authorizer(args, request.headers,
                          request.cookies, request.host, request,
                          forced_domain, scopes_to_check, simulated)

    def __init__(self, args={}, headers={}, cookies={}, host='',
                 request=None, forced_domain=None, scopes_to_check=[], simulated=False):
        self.query_arguments = args
        self.headers = headers
        self.cookies = cookies
        self.host = host
        self.request = request  # mainly for logging
        self.forced_domain = forced_domain
        self.scopes_to_check = scopes_to_check
        self.simulated = simulated

    def get_required_defaults(self):
        return []

    def get_optional_defaults(self):
        return ['cert', 'token', 'cookies']

    def parse_steps(self):
        idm_role = self.query_arguments.get('idm_role', '')
        fields_data = self.query_arguments.get('fields_data', '')
        put_to_context('role', {'path': idm_role, 'fields_data': fields_data})
        if fields_data:
            try:
                fields_data = json.loads(fields_data)
                assert isinstance(fields_data, dict) or fields_data is None
            except (json.JSONDecodeError, AssertionError):
                return None

        idm_info = (idm_role, fields_data)
        required_steps = self.query_arguments.get('required')
        optional_steps = self.query_arguments.get('optional')
        if not optional_steps:
            if not required_steps:
                optional_steps = self.get_optional_defaults()
            else:
                optional_steps = []
        else:
            optional_steps = optional_steps.split(',')

        if not required_steps:
            required_steps = self.get_required_defaults()
        else:
            required_steps = required_steps.split(',')

        if all(step in webauth_step.LOGIN_STEPS for step in itertools.chain(required_steps, optional_steps)):
            return required_steps, optional_steps, idm_info
        else:
            return None

    @gen.coroutine
    def check_steps(self, steps, user_ip, strict, reasons, user_info):
        success = True if strict else None
        step_checkers = [webauth_step.LOGIN_STEPS[step](self, user_ip, self.forced_domain, self.scopes_to_check)
                         for step in steps]
        results = yield [step.check() for step in step_checkers]
        for step, (status, info) in zip(steps, results):
            if status is True:
                success = True
                if user_info is not None and user_info != info:
                    reasons.append('Logins from different sources are not equal to each other')
                    success = False
                    break
                elif user_info is None:
                    user_info = info
            else:
                reasons.append('Checking %s: %s' % (step, info))
                if status is False or (status is None and strict):
                    success = False
                    break
        if not steps:
            success = True
        if success:
            raise gen.Return((True, user_info))
        else:
            raise gen.Return((False, reasons))

    def get_user_ip(self):
        header = self.headers.get(b'X-Forwarded-For')
        if not header:
            return ''
        return get_user_ip(header)

    @gen.coroutine
    def get(self):
        userIp = self.get_user_ip()

        # authorizer can work correctly only if we have X-Forward-For-Y
        # so if it is empty - it would be better to return 403
        if not userIp:
            raise gen.Return((False, (['Cannot determine IP'],)))

        query_info = self.parse_steps()
        if query_info:
            required_steps, optional_steps, idm_info = query_info
        else:
            raise gen.Return((False, (['Incorrect query'],)))

        user_info = None
        reasons = []
        for steps, strict in (required_steps, True), (optional_steps, False):
            status, info = yield self.check_steps(steps, userIp, strict, reasons, user_info)
            if not status:
                raise gen.Return((False, (info,)))
            user_info = info

        login, uid = user_info
        idm_role, fields_data = idm_info
        if idm_role:
            cache_start = time.time()
            idm_role_status = yield role_cache.check_role_cache(login, idm_role, fields_data)
            request_ms = round(time.time() - cache_start, 4) * 1000
            logger.info('Role request to redis: %f ms', request_ms)

            if not idm_role_status:
                error_message = 'Checking idm role: system, node or role do not exist'
                raise gen.Return((False, ([error_message], login, uid)))

        raise gen.Return((True, (login, uid)))
