# encoding: utf-8
from __future__ import unicode_literals

import datetime
import logging
import os

import library.python.resource
from statface_client import StatfaceClient, constants
from yql.api.v1.client import YqlClient

from intranet.webauth.lib.settings import WEBAUTH_YQL_TOKEN, WEBAUTH_STATFACE_TOKEN

logger = logging.getLogger('balancer_request_time')


class BalancerRequestTimeMonitoring(object):
    yql_script_name = 'balancer_request_time.yql'
    report_path = 'ExtData/webauth/balancer_request_time'
    time_shift = datetime.timedelta(minutes=25)
    timedelta = datetime.timedelta(minutes=60)
    timezone_offset = datetime.timedelta(hours=3)

    def abs_path(self, file_name):
        base_path = os.path.dirname(os.path.realpath(__file__))
        return os.path.join(base_path, 'yql_attach', file_name)

    def format_date(self, date):
        return (date - datetime.timedelta(minutes=date.minute % 5, seconds=date.second)).strftime('%Y-%m-%dT%H:%M:%S')

    def reformat_data(self, data):
        result_data = dict()
        for row in data:
            row = row.copy()
            component = '_'.join(row.pop('component').split('-'))
            for slug, value in row.items():
                result_data['_'.join([component, slug])] = value
            result_data['_'.join([component, 'rps'])] = float(row['requests_number']) / self.timedelta.seconds
        return result_data

    def get_yql_data(self, **kwargs):
        client = YqlClient(token=WEBAUTH_YQL_TOKEN)
        query = library.python.resource.find(self.yql_script_name)
        query = query.format(**kwargs)

        request = client.query(query, syntax_version=1)
        request.run()
        if request.status == 'ERROR':
            errors_digest = '; '.join([error.message for error in request.errors])
            logger.info('balancer_request_time: Yql error: %s', errors_digest)
            raise RuntimeError(errors_digest)

        for table in request.get_results():
            table.fetch_full_data()
            column_names = [col[0] for col in table.columns]

            for row in table.rows:
                yield dict(zip(column_names, row))

    def save_data(self, data, stat_host):
        stf_client = StatfaceClient(oauth_token=WEBAUTH_STATFACE_TOKEN, host=stat_host)
        report = stf_client.get_report(self.report_path)
        report.upload_data(data=data, scale='minutely')

    def run(self):
        to_date = datetime.datetime.now() + self.timezone_offset - self.time_shift
        from_date = to_date - self.timedelta
        if os.getenv('YENV_TYPE') == 'production':
            stat_host = constants.STATFACE_PRODUCTION
            components = ['check-oauth-token', 'app-plus', 'save-oauth-token', 'oauth-callback']
            balancer_names = '("tools-webauth-int", "tools-webauth")'
            environments = '("production", "production-ext")'
        else:
            stat_host = constants.STATFACE_BETA
            components = ['check-oauth-token', 'auth-request', 'save-oauth-token', 'oauth-callback']
            balancer_names = '("tools-int-test")'
            environments = '("testing")'

        logger.info('balancer_request_time: starting yql query')
        self.raw_data = self.get_yql_data(
            from_date=self.format_date(from_date),
            to_date=self.format_date(to_date),
            component_re="({})".format("|".join(components)),
            balancer_names=balancer_names,
            environments=environments,
        )
        data = self.reformat_data(self.raw_data)
        data['fielddate'] = self.format_date(to_date)
        logger.info('balancer_request_time: uploading data')
        self.save_data(data, stat_host)


def main():
    BalancerRequestTimeMonitoring().run()


if __name__ == '__main__':
    main()
