# coding: utf-8
from __future__ import unicode_literals

import json
import os
import time

import tornadis
from tornado import gen

from intranet.webauth.lib.monitorings.base import BaseMonitoringHandler
from intranet.webauth.lib.role_cache import (
    get_cache_version,
    get_version_timestamp,
    is_version_nonempty,
)
from intranet.webauth.lib.settings import (
    WEBAUTH_IDM_CACHE_FOLDER,
    WEBAUTH_IDM_CACHE_TTL_FOR_MONITORING,
)


class CheckCacheIntegrityHandler(BaseMonitoringHandler):
    @gen.coroutine
    def monitoring(self):
        if not os.path.exists(WEBAUTH_IDM_CACHE_FOLDER):
            raise gen.Return('IDM cache folder not exists')
        errors = []

        webauth_idm_systems = []

        read_path = os.path.join(WEBAUTH_IDM_CACHE_FOLDER, "webauth_idm_systems.json")
        if not os.path.exists(read_path):
            errors.append('Cache file for Webauth IDM systems does not exists')
        elif os.path.getsize(read_path) == 0:
            errors.append('Cache file for Webauth IDM systems is empty')
        else:
            webauth_idm_systems = json.load(open(read_path, 'r'))

        for system in webauth_idm_systems:
            if not system:
                continue
            output_path = os.path.join(WEBAUTH_IDM_CACHE_FOLDER, "%s.json" % system)
            if not os.path.exists(output_path):
                errors.append('Cache file for %s not exists' % system)
                continue
            if os.path.getsize(output_path) == 0:
                errors.append('Cache file for %s is empty' % system)
        raise gen.Return(errors)


class CheckCacheFreshnessHandler(BaseMonitoringHandler):
    @gen.coroutine
    def monitoring(self):
        errors = []

        webauth_idm_systems = []

        read_path = os.path.join(WEBAUTH_IDM_CACHE_FOLDER, "webauth_idm_systems.json")
        if not os.path.exists(read_path):
            errors.append('Cache file for Webauth IDM systems does not exists')
        elif os.path.getsize(read_path) == 0:
            errors.append('Cache file for Webauth IDM systems is empty')
        else:
            webauth_idm_systems = json.load(open(read_path, 'r'))

        for system in webauth_idm_systems:
            if not system:
                continue
            output_path = os.path.join(WEBAUTH_IDM_CACHE_FOLDER, "%s.json" % system)
            if not os.path.exists(output_path):
                errors.append('Cache file for %s not exists' % system)
                continue
            if time.time() - os.path.getmtime(output_path) > WEBAUTH_IDM_CACHE_TTL_FOR_MONITORING:
                errors.append('Cache file for %s expired' % system)

        try:
            current_version = yield get_cache_version()
            if current_version is None:
                errors.append('There is no roles in Redis')
            else:
                version_timestamp = yield get_version_timestamp(current_version)
                if version_timestamp is None:
                    errors.append('No timestamp set for current version')
                else:
                    if time.time() - version_timestamp > WEBAUTH_IDM_CACHE_TTL_FOR_MONITORING:
                        errors.append('Redis cache expired')
        except tornadis.TornadisException as err:
            errors.append('Could not connect to redis: {}'.format(err))

        raise gen.Return(errors)


class CheckRedisIntegrityHandler(BaseMonitoringHandler):
    @gen.coroutine
    def monitoring(self):
        try:
            version = yield get_cache_version()
            if version is None:
                raise gen.Return('There is no roles in Redis')
            is_nonempty = yield is_version_nonempty(version)
            if not is_nonempty:
                raise gen.Return('Current cache version is empty')
        except tornadis.TornadisException as err:
            raise gen.Return('Could not connect to redis: {}'.format(err))
