# encoding: utf-8
from __future__ import unicode_literals

import json
import logging
import time
from urllib import urlencode, quote, unquote
from urlparse import urlparse

from tornado import gen, httpclient, web
from ylog.context import log_context

from intranet.webauth.lib import settings
from intranet.webauth.lib.crypto_utils import encrypt_and_sign
from intranet.webauth.lib.utils import request_log_context

logger = logging.getLogger(__name__)


class OAuthCallbackHandler(web.RequestHandler):
    # disables response caching
    def compute_etag(self):
        return None

    @gen.coroutine
    def get(self):
        with log_context(**request_log_context(self.request)):
            logger.info('Incoming request')
            code = self.get_query_argument('code', default='')
            state = self.get_query_argument('state', default=':')
            try:
                original_url = unquote(state.split(':')[0])
                original_host = urlparse(original_url).netloc
            except Exception:
                self.set_status(403)
                self.finish('Incorrect "state" argument format')
                raise gen.Return()

            body = {
                'grant_type': 'authorization_code',
                'code': code,
                'client_id': settings.WEBAUTH_OAUTH_APPLICATION_ID,
                'client_secret': settings.WEBAUTH_OAUTH_APPLICATION_SECRET,
            }
            request = httpclient.HTTPRequest(
                "https://oauth.yandex-team.ru/token",
                method='POST',
                body=urlencode(body),
                ca_certs=settings.WEBAUTH_ROOT_CERT_LOCATION,
                validate_cert=settings.WEBAUTH_HTTP_CLIENT_VALIDATE_SSL,
            )
            try:
                response = yield httpclient.AsyncHTTPClient().fetch(request)
                token = json.loads(response.body).get('access_token', '')
            except Exception as err:
                response_body = ''
                if isinstance(err, httpclient.HTTPError):
                    response = getattr(err, 'response', None)
                    response_body = getattr(response, 'body', '')
                logger.warning('Exception while fetching token from OAuth: %s, response="%s"', err, response_body)
                self.set_status(403)
                self.finish('Invalid response from OAuth')
                raise gen.Return()
            if not token:
                logger.warning('Invalid response while fetching token from OAuth: %s', response.body)
                self.set_status(403)
                self.finish('No or invalid access token')
                raise gen.Return()

            timestamp_and_token = '{timestamp}:{token}'.format(timestamp=int(time.time()),
                                                               token=quote(token))
            enc_token = encrypt_and_sign(timestamp_and_token)
            url = "https://%s/__webauth_save_oauth_token?token=%s&state=%s" % (original_host,
                                                                               quote(enc_token),
                                                                               quote(state))
            self.redirect(url)
