# encoding: utf-8
from __future__ import unicode_literals

import logging
import re
import time
from urllib import unquote, quote
from urlparse import urlparse

import tornado.web
from tornado import gen
from ylog.context import log_context

from intranet.webauth.lib import settings
from intranet.webauth.lib.crypto_utils import (
    get_hmac,
    decrypt_signed_value,
    encrypt_and_sign,
)
from intranet.webauth.lib.utils import request_log_context

logger = logging.getLogger(__name__)


class SaveOAuthTokenHandler(tornado.web.RequestHandler):
    # disables response caching
    def compute_etag(self):
        return None

    def decline(self, reason):
        self.set_status(403)
        self.finish(reason)

    @gen.coroutine
    def get(self):
        with log_context(**request_log_context(self.request)):
            logger.info('Incoming request')

        enc_token = self.get_query_argument('token', '').strip()
        timestamp_and_token = decrypt_signed_value(enc_token)
        if timestamp_and_token is None:
            self.decline('Invalid token')
            raise gen.Return()
        timestamp, token = timestamp_and_token.split(':')
        token = unquote(token)
        if time.time() - int(timestamp) > 30:
            self.decline('Token expired')
            raise gen.Return()
        try:
            original_url, state = self.get_query_argument('state', ':').split(':')
        except Exception:
            self.decline('Incorrect "state" argument format')
            raise gen.Return()
        original_url = unquote(original_url)
        state = unquote(state)

        csrf_token = self.get_cookie(settings.WEBAUTH_CSRF_TOKEN_COOKIE, default=None)
        if csrf_token is None:
            self.decline('No csrf_token')
            raise gen.Return()
        if state != get_hmac(csrf_token):
            self.decline('Incorrect state parameter')
            raise gen.Return()

        netloc = urlparse(original_url).netloc
        pattern = re.compile(settings.CROWDTEST_ZONE_PATTERN)
        match = pattern.search(netloc)
        domain = '.' + match.group().lstrip('.') if match else None

        encrypted_token = quote(encrypt_and_sign(token))

        self.set_cookie(
            name=settings.WEBAUTH_CLIENT_TOKEN_COOKIE,
            value=encrypted_token,
            domain=domain,
            secure=True,
            httponly=True,
            samesite='None',
        )
        self.clear_cookie(settings.WEBAUTH_CSRF_TOKEN_COOKIE)
        self.redirect(original_url)
