# encoding: utf-8
from __future__ import unicode_literals

import json
import logging
import re
import time
import urllib
import urlparse

import tornadis
from tornado import gen, httpclient

from intranet.webauth.lib.crypto_utils import get_hash
from intranet.webauth.lib.settings import (
    WEBAUTH_YANDEX_CERT_DOMAINS,
    WEBAUTH_YANDEX_CERT_ISSUERS,
    WEBAUTH_USER_CREDENTIALS_CACHE_TTL,
    WEBAUTH_HTTP_POOL_SIZE,
)
from intranet.webauth.lib.utils import (
    get_redis_pool,
    fetch_and_retry_timeouts,
)
from intranet.webauth.lib import settings

logger = logging.getLogger(__name__)

INTERNAL_ZONES = ['.yandex-team.ru']
BLACKBOX_INTERNAL_API = 'http://blackbox-ipv6.yandex-team.ru/blackbox'
BLACKBOX_EXTERNAL_API = 'http://blackbox.yandex.net/blackbox'
BLACKBOX_SECURE_PARAMS = ['sslsessionid', 'sessionid', 'oauth_token']
BLACKBOX_TIMEOUT = 0.05  # 50 ms
BLACKBOX_ATTEMPTS = 2

httpclient.AsyncHTTPClient.configure('tornado.curl_httpclient.CurlAsyncHTTPClient', max_clients=WEBAUTH_HTTP_POOL_SIZE)


def secure_url(url):
    result = list(urlparse.urlparse(url))  # [scheme, netloc, path, params, query, fragment]
    query = urlparse.parse_qs(result[4])
    for key in query:
        if key in BLACKBOX_SECURE_PARAMS:
            query[key] = '<safe>'
        else:
            query[key] = query[key][0]  # always a 1-element list
    result[4] = urllib.unquote(urllib.urlencode(query))
    return urlparse.ParseResult(*result).geturl()


class Step(object):
    # forced_domain - [None, 'internal', 'external']
    # It is used for assessor authorization scheme when external domains want to authenticate internal users
    def __init__(self, authorizer, user_ip, forced_domain=None, scopes_to_check=[]):
        self.authorizer = authorizer
        self.user_ip = user_ip
        self.cookies = {cookie: self.authorizer.cookies[cookie].value for cookie in self.authorizer.cookies}
        self.forced_domain = forced_domain
        self.scopes_to_check = scopes_to_check

    @gen.coroutine
    def check(self):
        raise NotImplementedError()


class BlackboxStep(Step):
    def is_internal_domain(self):
        assert self.forced_domain in ('internal', 'external', None)
        if self.forced_domain == 'internal':
            return True
        elif self.forced_domain == 'external':
            return False
        else:
            is_internal = any(self.authorizer.host.endswith(zone) for zone in INTERNAL_ZONES)
            return is_internal

    @gen.coroutine
    def make_auth_query(self, token=None, scopes=[], session_id=None, session_id2=None):
        query = {'userip': self.user_ip, 'format': 'json', 'dbfields': 'accounts.login.uid'}
        if token:
            query['method'] = 'oauth'
#            query['get_user_ticket'] = 'yes'
            query['oauth_token'] = token
            if scopes:
                query['scopes'] = ','.join(scopes)
        if session_id:
            query['method'] = 'sessionid'
#            query['get_user_ticket'] = 'yes'
            query['sessionid'] = session_id
            query['host'] = 'yandex-team.ru'
            if session_id2:
                query['sslsessionid'] = session_id2

        blackbox_api = BLACKBOX_INTERNAL_API if self.is_internal_domain() else BLACKBOX_EXTERNAL_API
        bb_url = blackbox_api + '?' + urllib.urlencode(query)
        request = httpclient.HTTPRequest(
            url=bb_url,
            connect_timeout=BLACKBOX_TIMEOUT,
            request_timeout=BLACKBOX_TIMEOUT,
            ca_certs=settings.WEBAUTH_ROOT_CERT_LOCATION,
            validate_cert=settings.WEBAUTH_HTTP_CLIENT_VALIDATE_SSL,
        )
        response = None
        try:
            http_client = httpclient.AsyncHTTPClient()
            request_start = time.time()
            response = yield fetch_and_retry_timeouts(http_client, request, attempts=BLACKBOX_ATTEMPTS)
        except Exception as err:
            logger.error('Bad blackbox response (%s):  %s', secure_url(bb_url), response if response else err)
            raise gen.Return((False, 'Bad blackbox answer'))
        finally:
            request_ms = round(time.time() - request_start, 4) * 1000
            request_type = 'token' if token else 'cookies'
            actual_request_ms = round(response.request_time, 4) * 1000 if response else request.request_timeout
            # No idea if it is accurate (asynchronous loop), but it will have to do
            logger.info('Request to blackbox: %s, %f ms(with eventloop), %f ms(request)', request_type, request_ms, actual_request_ms)
        try:
            data = json.loads(response.body)
        except ValueError:
            logger.error('Could not parse blackbox answer (%s): %s', secure_url(bb_url), response)
            raise gen.Return((False, 'Could not parse blackbox answer'))

        raise gen.Return((True, data))

    def get_cache_key(self, token=None, token_scopes=None, session_id=None, session_id2=None):
        location = 'internal' if self.is_internal_domain() else 'external'
        key_pattern = 'credentials/{type}/{location}/{scopes}/{value}'
        if token:
            scopes_string = '|'.join(sorted(token_scopes)) if token_scopes else '<NONE>'
            key = key_pattern.format(
                type='token',
                location=location,
                scopes=scopes_string,
                value=get_hash(token),
            )
        elif session_id:
            session_id2 = session_id2 or '<NONE>'  # отсутствие второй куки - не приговор
            key = key_pattern.format(
                type='cookies',
                location=location,
                scopes='<NONE>',
                value=get_hash(session_id + '###' + session_id2),
            )
        else:
            key = None
        return key

    @gen.coroutine
    def read_cache(self, token=None, token_scopes=None, session_id=None, session_id2=None):
        key = self.get_cache_key(token=token, token_scopes=token_scopes, session_id=session_id, session_id2=session_id2)
        if key is None:
            raise gen.Return()

        with (yield get_redis_pool().connected_client()) as client:
            if isinstance(client, tornadis.TornadisException):
                raise gen.Return(None)
            raw_result = yield client.call('GET', key)
        if isinstance(raw_result, tornadis.TornadisException):
            raise gen.Return(None)

        try:
            result = tuple(json.loads(raw_result))
        except Exception:
            result = None
        raise gen.Return(result)

    @gen.coroutine
    def write_cache(self, value, token=None, token_scopes=None, session_id=None, session_id2=None):
        key = self.get_cache_key(token=token, token_scopes=token_scopes, session_id=session_id, session_id2=session_id2)
        if key is None:
            raise gen.Return(False)
        value_string = json.dumps(value)

        with (yield get_redis_pool().connected_client()) as client:
            if isinstance(client, tornadis.TornadisException):
                raise gen.Return(False)
            result = yield client.call('SETEX', key, WEBAUTH_USER_CREDENTIALS_CACHE_TTL, value_string)

        if isinstance(result, tornadis.TornadisException):
            raise gen.Return(False)
        else:
            raise gen.Return(True)

    @staticmethod
    def check_credentials(response):
        credentials_status = response.get('status', {}).get('value')
        return credentials_status in ['VALID', 'NEED_RESET']

    @staticmethod
    def get_user_info(response):
        return (response.get('dbfields', {}).get('accounts.login.uid'),
                response.get('uid', {}).get('value'))

    @staticmethod
    def check_security(response):
        security_status = response.get('auth', {}).get('secure', False)
        return security_status


class TokenStep(BlackboxStep):
    @gen.coroutine
    def check(self):
        auth_header = self.authorizer.headers.get('Webauth-Authorization')
        if auth_header is None:
            auth_header = self.authorizer.headers.get('Authorization')
        oauth_token = None
        if auth_header:
            match = re.match(r'OAuth (?P<token>.+)', auth_header)
            if match is not None:
                oauth_token = match.group('token')

        if not oauth_token:
            raise gen.Return((None, 'No OAuth token provided'))

        # Смотрим в кеш
        cache_duration = 0.0
        cache_start = time.time()
        user_info = yield self.read_cache(token=oauth_token, token_scopes=self.scopes_to_check)
        cache_duration += time.time() - cache_start
        if user_info:
            request_ms = round(cache_duration, 4) * 1000
            logger.info('Auth request to redis: %s, %f ms', 'token', request_ms)
            raise gen.Return((True, user_info))

        # Ничего не нашли в кеше, по-честному идём в ЧЯ
        status, info = yield self.make_auth_query(token=oauth_token, scopes=self.scopes_to_check)
        if not status:
            request_ms = round(cache_duration, 4) * 1000
            logger.info('Auth request to redis: %s, %f ms', 'token', request_ms)
            raise gen.Return((status, info))
        if not self.check_credentials(info):
            request_ms = round(cache_duration, 4) * 1000
            logger.info('Blackbox response: %s', info)
            logger.info('Auth request to redis: %s, %f ms', 'token', request_ms)
            raise gen.Return((False, 'Can not validate token via blackbox'))

        user_info = self.get_user_info(info)

        # Записываем свежачок в кеш
        cache_start = time.time()
        yield self.write_cache(user_info, token=oauth_token, token_scopes=self.scopes_to_check)
        cache_duration += time.time() - cache_start
        request_ms = round(cache_duration, 4) * 1000
        logger.info('Auth request to redis: %s, %f ms', 'token', request_ms)

        raise gen.Return((True, user_info))


class CookiesStep(BlackboxStep):
    @gen.coroutine
    def check(self):
        session_id = self.cookies.get('Session_id')
        session_id2 = self.cookies.get('sessionid2')
        if not session_id:
            raise gen.Return((None, 'No SessionID in cookies'))

        # Смотрим в кеш
        cache_duration = 0.0
        cache_start = time.time()
        user_info = yield self.read_cache(session_id=session_id, session_id2=session_id2)
        cache_duration += time.time() - cache_start
        if user_info:
            request_ms = round(cache_duration, 4) * 1000
            logger.info('Auth request to redis: %s, %f ms', 'cookies', request_ms)
            raise gen.Return((True, user_info))

        # Ничего не нашли в кеше, по-честному идём в ЧЯ
        status, info = yield self.make_auth_query(session_id=session_id, session_id2=session_id2)

        if not status:
            request_ms = round(cache_duration, 4) * 1000
            logger.info('Auth request to redis: %s, %f ms', 'cookies', request_ms)
            raise gen.Return((status, info))
        if not self.check_credentials(info):
            request_ms = round(cache_duration, 4) * 1000
            logger.info('Blackbox response: %s', info)
            logger.info('Auth request to redis: %s, %f ms', 'cookies', request_ms)
            raise gen.Return((False, 'Can not validate cookies via blackbox'))
        if not self.check_security(info):
            request_ms = round(cache_duration, 4) * 1000
            logger.info('Auth request to redis: %s, %f ms', 'cookies', request_ms)
            raise gen.Return((False, 'Can not validate secure cookie via blackbox'))

        user_info = self.get_user_info(info)

        # Записываем свежачок в кеш
        cache_start = time.time()
        yield self.write_cache(user_info, session_id=session_id, session_id2=session_id2)
        cache_duration += time.time() - cache_start
        request_ms = round(cache_duration, 4) * 1000
        logger.info('Auth request to redis: %s, %f ms', 'cookies', request_ms)

        raise gen.Return((True, user_info))


class CertificateStep(Step):
    @gen.coroutine
    def get_uid_by_login(self, login):
        query = {'userip': self.user_ip, 'format': 'json', 'method': 'userinfo', 'login': login}

        is_internal = any(self.authorizer.host.endswith(zone) for zone in INTERNAL_ZONES)
        if self.forced_domain == 'internal' or self.forced_domain is None and is_internal:
            blackbox_api = BLACKBOX_INTERNAL_API
        elif self.forced_domain == 'external' or self.forced_domain is None and not is_internal:
            blackbox_api = BLACKBOX_EXTERNAL_API
        else:
            logger.error('Incorrect value in Step instance: forced_domain=%s', self.forced_domain)
            raise gen.Return((False, 'Internal error'))

        bb_url = blackbox_api + '?' + urllib.urlencode(query)
        request = httpclient.HTTPRequest(url=bb_url, connect_timeout=BLACKBOX_TIMEOUT, request_timeout=BLACKBOX_TIMEOUT)
        response = None
        try:
            http_client = httpclient.AsyncHTTPClient()
            response = yield http_client.fetch(request)
        except Exception:
            if response:
                logger.error('Bad blackbox response (%s):  %s', secure_url(bb_url), response)
            raise gen.Return(None)
        try:
            data = json.loads(response.body)
        except ValueError:
            logger.error('Could not parse blackbox answer (%s): %s', secure_url(bb_url), response)
            raise gen.Return(None)

        raise gen.Return(data.get('id'))

    def get_cn(self, field):
        cert_header = self.authorizer.headers.get(field)
        sep = ','
        # old nginx version uses slashes instead of commas
        if '/' in cert_header:
            sep = '/'
        try:
            # finding substring of type "CN=..."
            substring = [x for x in cert_header.split(sep) if x.startswith('CN=')][0]
            return substring.lstrip('CN=')
        except Exception:
            pass

        return None

    @gen.coroutine
    def check(self):
        cert_status = self.authorizer.headers.get('X-Qloud-Ssl-Verified', 'NONE')
        if cert_status == 'SUCCESS':
            issuer = self.get_cn('X-Qloud-Ssl-Issuer')
            if issuer is None:
                raise gen.Return((False, 'Incorrect SSL issuer format'))
            if issuer not in WEBAUTH_YANDEX_CERT_ISSUERS:
                raise gen.Return((False, 'Incorrect SSL issuer'))

            try:
                subject = self.get_cn('X-Qloud-Ssl-Subject')
                login, domain = subject.split('@')
            except Exception:
                raise gen.Return((False, 'Incorrect SSL subject format'))
            if domain not in WEBAUTH_YANDEX_CERT_DOMAINS:
                raise gen.Return((False, 'Incorrect SSL domain'))

            uid = yield self.get_uid_by_login(login)
            if not uid:
                raise gen.Return((False, 'Cannot get user uid using login'))
            raise gen.Return((True, (login, uid)))

        elif cert_status == 'FAILED':
            raise gen.Return((False, 'Cannot validate client certificate'))
        else:
            # NONE or false
            # the latter is kind of dangerous:
            # it means invalid certificate for deprecated routers,
            # but it is also set when Qloud is not configured to request certificate
            raise gen.Return((None, 'No client certificate provided'))


LOGIN_STEPS = {"cert": CertificateStep,
               "cookies": CookiesStep,
               "token": TokenStep}
