# encoding: utf-8
from __future__ import unicode_literals

import logging
import time

import requests

from intranet.webauth.lib.settings import (
    WEBAUTH_OAUTH_TOKEN,
    WEBAUTH_IDM_HOST,
    WEBAUTH_QLOUD_SYNC_MARK_FILE,
)
from intranet.webauth.lib.utils import (
    setup_logging,
    ylock_manager,
    requests_with_retry_policy,
    touch_file,
)

setup_logging()
logger = logging.getLogger('sync_qloud_idm')

TIMEOUT = 100
TRIES = 6
CA_CERTIFICATES_PATH = '/etc/ssl/certs/ca-certificates.crt'

IDM_API = '/api/v1'
IDM_HEADERS = {
    b'Content-Type': 'application/json',
    b'Authorization': 'OAuth %s' % WEBAUTH_OAUTH_TOKEN
}
IDM_SYSTEM = 'webauth-qloud'
IDM_PAGE_SIZE = 250

QLOUD_INSTALLATIONS = {
    'qloud-ext': 'platform.yandex-team.ru',
    'qloud': 'qloud.yandex-team.ru',
    # 'qloud-test': 'qloud-test.yandex-team.ru',
}
QLOUD_HEADERS = {"Authorization": "OAuth %s" % WEBAUTH_OAUTH_TOKEN}
QLOUD_PROJECT_TREE_API = "/api/project-tree"
QLOUD_ENVS_REMOVAL_THRESHOLD = 0.3  # 30% of what we have in IDM


def idm_to_qloud(node):
    installation, environment = node.strip('/').split('/', 1)
    environment = environment.replace('/', '.')
    return installation, environment


def qloud_to_idm(installation, environment):
    project, application, environment = environment.split('.')
    slugs = [installation, project, application, 'envs', environment]
    return '/%s/' % '/'.join(slugs)


def node_is_environment(node_path, with_keys=False):
    if node_path[0] != '/' or node_path[-1] != '/':
        return False
    node_path = node_path.strip('/')
    if with_keys:
        return node_path.count('/') == 9  # key/installation/key/project/key/application/key/role_type/key/environment
    else:
        return node_path.count('/') == 4  # installation/project/application/role_type/environment


def value_to_slug_path(value_path):
    slugs = value_path.strip('/').split('/')
    keys = ['installation', 'project', 'application', 'role_type', 'environment', 'role']
    result = ['']
    for key, slug in zip(keys, slugs):
        result.append(key)
        result.append(slug)
    result.append('')
    return '/'.join(result)


def add_nodes_for_environment(env_name, node_set):
    subrequests = []

    installation, project, application, _, environment = env_name.strip('/').split('/')
    nodes = [('/installation/', installation, installation.capitalize(), installation.capitalize()),
             ('/installation/%s/' % installation, 'project', 'Project', 'Проект'),
             ('/installation/%s/project/' % installation, project, project.capitalize(), project.capitalize()),
             ('/installation/%s/project/%s/' % (installation, project),
              'application', 'Application', 'Приложение'),
             ('/installation/%s/project/%s/application/' % (installation, project),
              application, application.capitalize(), application.capitalize()),
             ('/installation/%s/project/%s/application/%s/' % (installation, project, application),
              'role_type', 'Type', 'Тип'),
             ('/installation/%s/project/%s/application/%s/role_type/' % (installation, project, application),
              'app_admin', 'Application Administrator', 'Администратор приложения'),
             ('/installation/%s/project/%s/application/%s/role_type/' % (installation, project, application),
              'envs', 'Environments', 'Окружения'),
             ('/installation/%s/project/%s/application/%s/role_type/envs/' % (installation, project, application),
              'environment', 'Environment', 'Окружение'),
             ('/installation/%s/project/%s/application/%s/role_type/envs/environment/' % (installation,
                                                                                          project,
                                                                                          application),
              environment, environment.capitalize(), environment.capitalize()),
             ('/installation/%s/project/%s/application/%s/role_type/envs/environment/%s/' %
              (installation, project, application, environment),
              'role', 'Role', 'Роль'),
             ('/installation/%s/project/%s/application/%s/role_type/envs/environment/%s/role/' % (installation,
                                                                                                  project,
                                                                                                  application,
                                                                                                  environment),
              'responsible', 'Responsible', 'Ответственный'),
             ('/installation/%s/project/%s/application/%s/role_type/envs/environment/%s/role/' % (installation,
                                                                                                  project,
                                                                                                  application,
                                                                                                  environment),
              'user', 'User', 'Пользователь'),
             ]

    nodes_to_create = [node for node in nodes if ('%s%s/' % (node[0], node[1])) not in node_set]
    for node in nodes_to_create:
        parent_slug, slug, en_name, ru_name = node
        body = {'system': IDM_SYSTEM,
                'parent': parent_slug,
                'slug': slug,
                'name': {'en': en_name, 'ru': ru_name},
                }
        subrequest = {'method': 'POST',
                      'path': 'rolenodes',
                      'body': body}
        subrequests.append(subrequest)

    url = 'https://%s%s/batch/' % (WEBAUTH_IDM_HOST, IDM_API)
    try:
        session = requests_with_retry_policy(TRIES)
        response = session.post(url,
                                json=subrequests,
                                headers=IDM_HEADERS,
                                timeout=TIMEOUT,
                                verify=CA_CERTIFICATES_PATH,)
    except requests.exceptions.RequestException:
        logger.exception("sync_qloud_idm: Exception while Fetching POST request (%s)", url)
        return False, None

    if response.status_code != 200:
        logger.error("sync_qloud_idm: Fetching POST request (%s) returned code %d (%s)" %
                     (url, response.status_code, response.text))
        return False, None

    new_nodes = {('%s%s/' % node[:2]) for node in nodes_to_create}
    return True, new_nodes


def delete_nodes_for_environment(environment):
    node = '/%s%s' % (IDM_SYSTEM, value_to_slug_path(environment))
    url = 'https://%s%s/rolenodes%s' % (WEBAUTH_IDM_HOST, IDM_API, node)
    try:
        session = requests_with_retry_policy(TRIES)
        response = session.delete(url, headers=IDM_HEADERS, timeout=TIMEOUT, verify=CA_CERTIFICATES_PATH)
    except requests.exceptions.RequestException:
        logger.exception("sync_qloud_idm: Exception while Fetching DELETE request (%s)", url)
        return None

    if response.status_code != 204:
        logger.error("sync_qloud_idm: Fetching DELETE request (%s) returned code %d (%s)" % (response.request.url,
                                                                                             response.status_code,
                                                                                             response.text))
        return False

    return True


def get_environments_from_idm():
    nodes = []
    environments = []
    uri = '%s/rolenodes/?system=%s&limit=%d' % (IDM_API, IDM_SYSTEM, IDM_PAGE_SIZE)
    while uri is not None:
        url = 'https://%s' % WEBAUTH_IDM_HOST + uri
        try:
            session = requests_with_retry_policy(TRIES)
            response = session.get(url, headers=IDM_HEADERS, timeout=TIMEOUT, verify=CA_CERTIFICATES_PATH)
        except requests.exceptions.RequestException:
            logger.exception("sync_qloud_idm: Exception while Fetching GET request (%s)", url)
            return None

        if response.status_code != 200:
            logger.error("sync_qloud_idm: Fetching GET request (%s) returned code %d (%s)" % (response.request.url,
                                                                                              response.status_code,
                                                                                              response.text))
            return None

        body = response.json()
        nodes += [node['slug_path'] for node in body['objects']]
        environments += [node['value_path']
                         for node in body['objects']
                         if node_is_environment(node['value_path']) and not node['is_key']]
        uri = body['meta']['next']
    return set(nodes), set(environments)


def get_environments_from_qloud():
    environments = []
    for installation_name, installation_host in QLOUD_INSTALLATIONS.items():
        url = "https://%s%s" % (installation_host, QLOUD_PROJECT_TREE_API)

        try:
            session = requests_with_retry_policy(TRIES)
            response = session.get(url, headers=QLOUD_HEADERS, verify=CA_CERTIFICATES_PATH, timeout=TIMEOUT)
        except requests.exceptions.RequestException:
            logger.exception("sync_qloud_idm: Exception while Fetching GET request (%s)", url)
            return None

        if response.status_code != 200:
            logger.error("sync_qloud_idm: Fetching GET request (%s) returned code %d (%s)" % (response.request.url,
                                                                                              response.status_code,
                                                                                              response.text))
            return None

        data = response.json()
        for project in data.get("projects"):
            for application in project.get("applications"):
                for environment in application.get("environments"):
                    id = environment.get("objectId")
                    if id:
                        environments.append(qloud_to_idm(installation_name, id))

    qloud_environments = set(environments)
    return qloud_environments


def sync_qloud_environments():
    start = time.time()

    with ylock_manager.lock('sync_qloud_idm', block=False) as locked:
        if not locked:
            logger.warning("sync_qloud_idm: could not acquire lock")
            exit(3)
        added = 0
        not_added = 0
        removed = 0
        not_removed = 0
        # acquiring qloud environments
        qloud_environments = get_environments_from_qloud()
        if qloud_environments is None:
            logger.error('sync_qloud_idm: Could not get Qloud envs')
            exit(1)
        # acquiring idm roles
        idm_return_value = get_environments_from_idm()
        if idm_return_value is None:
            logger.error('sync_qloud_idm: Could not get IDM nodes')
            exit(2)
        idm_nodes, idm_environments = idm_return_value

        envs_to_add = qloud_environments - idm_environments
        envs_to_remove = idm_environments - qloud_environments
        removal_threshold = int(QLOUD_ENVS_REMOVAL_THRESHOLD * len(idm_environments))
        to_remove = len(envs_to_remove)
        if to_remove > removal_threshold:
            logger.error(
                'Threshold reached while removing environments from IDM: %d > %d',
                to_remove,
                removal_threshold,
            )
            exit(4)

        # adding roles for new environments:
        for i, env in enumerate(envs_to_add):
            logger.info('sync_qloud_idm: ADD %s (%d/%d)', env, i, len(envs_to_add))
            result, new_nodes = add_nodes_for_environment(env, idm_nodes)
            if not result:
                not_added += 1
                logger.error('sync_qloud_idm: %s: FAILED to add nodes', env)
            else:
                idm_nodes |= new_nodes
                added += 1
        # deleting roles for deleted environments:
        for i, env in enumerate(envs_to_remove):
            logger.info('sync_qloud_idm: REMOVE %s (%d/%d)', env, i, len(envs_to_remove))
            result = delete_nodes_for_environment(env)
            if not result:
                not_removed += 1
                logger.error('sync_qloud_idm: %s: FAILED to delete nodes', env)
            else:
                removed += 1
        finish = time.time()
        duration = int(finish - start)
        mins, secs = divmod(duration, 60)

        touch_file(WEBAUTH_QLOUD_SYNC_MARK_FILE)

        logger.info('sync_qloud_idm: successfully finished sync (%dm%ds): '
                    '%d envs added, %d not added (errors), %d removed, %d not removed',
                    mins,
                    secs,
                    added,
                    not_added,
                    removed,
                    not_removed,)


def main():
    sync_qloud_environments()


if __name__ == '__main__':
    main()
