# encoding: utf-8
from __future__ import unicode_literals

import fcntl
import logging
import logging.config
import os

import requests
import requests.adapters
import tornadis
import urllib3
import ylock
from tornado import gen
from tornado.httpclient import HTTPError

from intranet.webauth.lib import settings

RSYSLOG_PORT = 514
HEADERS_TO_STRIP = ('Authorization', 'Webauth-Authorization', 'Cookie')

ylock_manager = ylock.backends.create_manager(**settings.WEBAUTH_YLOCK_CONFIG)

_redis_instance = None


def get_redis_pool():
    global _redis_instance
    if not _redis_instance:
        _redis_instance = tornadis.ClientPool(
            host=settings.WEBAUTH_REDIS_HOST,
            port=settings.WEBAUTH_REDIS_PORT,
            max_size=settings.WEBAUTH_REDIS_POOL_SIZE,
            connect_timeout=settings.WEBAUTH_REDIS_CONNECT_TIMEOUT,
            read_timeout=settings.WEBAUTH_REDIS_READ_TIMEOUT,
        )
    return _redis_instance


def setup_logging():
    level = 'DEBUG' if settings.WEBAUTH_DEBUG else 'INFO'
    lib_level = 'WARNING' if not settings.WEBAUTH_DEBUG else 'DEBUG'
    config = {
        'version': 1,
        'formatters': {
            'plain': {'format': '%(asctime)s: %(levelname)s: %(pathname)s: %(message)s'},
            'json': {'()': 'ylog.format.QloudJsonFormatter'},
        },
        'handlers': {
            'stderr': {
                'level': 'INFO',
                'formatter': 'json',
                'class': 'logging.StreamHandler',
                'stream': 'ext://sys.stderr'
            },
        },
        'loggers': {
            '': {
                'handlers': ['stderr'],
                'level': level,
                'propagate': True
            },
            'requests': {
                'handlers': ['stderr'],
                'level': lib_level,
                'propagate': True
            },
            'urllib3': {
                'handlers': ['stderr'],
                'level': lib_level,
                'propagate': True
            },

        },
        'propagate': True,
    }
    if settings.YENV_TYPE == 'development':
        config['handlers']['stderr']['formatter'] = 'plain'

    logging.config.dictConfig(config)

    for logger in logging.Logger.manager.loggerDict.values():
        if isinstance(logger, logging.Logger) and logger.name.startswith('intranet.webauth.lib.'):
            logger.disabled = 0
            logger.setLevel(level)


def get_user_ip(forwarded_for):
    ips = [ip.strip() for ip in forwarded_for.split(',') if ip.strip()]
    return ips[0] if ips else ''


def request_log_context(authorizer):
    headers = dict(authorizer.headers)
    ip = get_user_ip(headers.get(b'X-Forwarded-For', ''))
    url = headers.pop('X-Original-Url', None)
    for header in HEADERS_TO_STRIP:
        if header in headers:
            headers[header] = '<STRIPPED>'

    cookies = {
        name: cookie.value
        for name, cookie in authorizer.cookies.items()
        if name != settings.WEBAUTH_CLIENT_TOKEN_COOKIE and not name.lower().startswith('session')
    }

    return {
        'simulated': getattr(authorizer, 'simulated', False),
        'ip': ip,
        'url': url,
        'query': authorizer.query_arguments,
        'headers': headers,
        'cookies': cookies
    }


def requests_with_retry_policy(tries, delay_block=1):
    max_retries = urllib3.Retry(
        total=tries - 1,
        backoff_factor=delay_block,
        status_forcelist=(400, 403, 404, 500, 502, 504),
    )
    session = requests.Session()
    session.mount('https://', requests.adapters.HTTPAdapter(max_retries=max_retries))
    return session


def touch_file(path):
    if os.path.exists(path):
        os.utime(path, None)
    else:
        open(path, 'w').close()


class lock_file(object):
    def __init__(self, path, blocking):
        self.lock_path = "%s.lock" % path
        self.lock_file = None
        self.blocking = blocking
        self.locked = False

    def __enter__(self):
        flags = fcntl.LOCK_EX
        if not self.blocking:
            flags |= fcntl.LOCK_NB
        try:
            self.lock_file = open(self.lock_path, 'w')
            fcntl.flock(self.lock_file, flags)
        except IOError:
            pass
        else:
            self.locked = True
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.locked:
            fcntl.flock(self.lock_file, fcntl.LOCK_UN)


# Can be modified to retry arbitrary errors
@gen.coroutine
def fetch_and_retry_timeouts(http_client, request, attempts=1, retry_wait=0.0, retry_sleep_multiplier=1.0, **kwargs):
    retriable_codes = (599,)  # 599 == no response (e.g. timeout)
    total_network_time = 0.0
    for attempt in range(attempts):
        try:
            response = yield http_client.fetch(request, **kwargs)
        except HTTPError as err:
            if err.response:
                total_network_time += err.response.request_time
            else:
                timeout = max(request.connect_timeout or 0.0, request.request_timeout or 0.0)
                if timeout:
                    total_network_time += timeout
            if err.code not in retriable_codes or attempt == attempts - 1:
                raise
            if retry_wait > 0:
                yield gen.sleep(retry_wait)
                retry_wait *= retry_sleep_multiplier
        else:
            total_network_time += response.request_time
            response.request_time = total_network_time  # summing up the time of all requests that were made
            raise gen.Return(response)


def patch_cookies():
    """
    Добавляет возможность задать в cookie SameSite.
    Такая возможность появилась только в python 3.8
    """
    try:
        from http.cookies import Morsel
    except ImportError:
        from Cookie import Morsel
    Morsel._reserved[str('samesite')] = str('SameSite')
