import string
import sys


# Donald Knuth iterative algorithm of computing inverse modulo
# http://www.di-mgt.com.au/euclidean.html#KNU298
def modinv(value, modulo):
    is_odd_iteration = True

    value1 = 1
    value3 = value
    modulo1 = 0
    modulo3 = modulo

    while modulo3 > 0:
        q = value3 // modulo3
        t3 = value3 % modulo3
        t1 = value1 + q * modulo1

        # swap
        value1 = modulo1
        modulo1 = t1
        value3 = modulo3
        modulo3 = t3
        is_odd_iteration = not is_odd_iteration

    if value3 != 1:
        raise Exception('modular inverse does not exist for ' + str(value))
    else:
        return value1 if is_odd_iteration else (modulo - value1)


_PRIME = 604462909807314587353111

BASE_32_DIGIT_MASK = (1 << 5) - 1

_BASE_32_DIGITS = string.digits + string.ascii_lowercase
_DIGIT_TO_INT = {_BASE_32_DIGITS[i]: i for i in range(0, 32)}
_SUFFIX_LEN = 16


def _to_base_32(number):
    result = ""
    while number != 0:
        digit = _BASE_32_DIGITS[number & BASE_32_DIGIT_MASK]
        number //= 32
        result = digit + result
    return result


def generate_tracker_id(cluster_id, connect_id):
    inverse = modinv(connect_id, _PRIME)
    suffix = _to_base_32(inverse)
    return cluster_id + "g" + suffix[:_SUFFIX_LEN].rjust(_SUFFIX_LEN, "0")


def tracker_id_to_connect_id(tracker_id):
    binary_str = ""
    for digit in tracker_id[4:]:
        binary_str = binary_str + "{0:b}".format(_DIGIT_TO_INT[digit]).rjust(5, "0")
    inversed_connect_id = int(binary_str, 2)
    return modinv(inversed_connect_id, _PRIME)


def render(connect_id):
    # bpf - PROD
    # aat - PREPROD
    tracker_id = generate_tracker_id("bpf", connect_id)
    connect_id_for_tracker = tracker_id_to_connect_id(tracker_id)
    print(str(connect_id) + " -> " + tracker_id + " -> " + str(connect_id_for_tracker))
    if connect_id != connect_id_for_tracker:
        raise Exception(str(connect_id) + " != " + str(connect_id_for_tracker))


def main():
    if len(sys.argv) < 2:
        print('trackerID required:\n   $ trackerid_to_dirid bpfg935ng1538m2dbvh1')
        return
    print(tracker_id_to_connect_id(sys.argv[1]))


if __name__ == "__main__":
    main()
