import errno
import os
import subprocess
import logging
import sys
import time
import tarfile
import tempfile
from datetime import datetime

import redis

from yatest.common.network import PortManager
from library.python.testing.recipe import declare_recipe, set_env

REDIS_TGZ = 'redis.tgz'


def log(msg):
    print(str(datetime.now()) + ' ' + msg)
    sys.stdout.flush()


def start(argv):
    if 'REDIS_RECIPE_PORT' in os.environ or 'REDIS_RECIPE_HOST' in os.environ:
        set_env('SKIP_CLEANUP_REDIS', '1')
        log('<< Using preconfigured REDIS >>')
        return

    log(f'Start redis - {time.time()}')

    tmp_dir = tempfile.gettempdir()
    log(f'Temp dir: {tmp_dir}')
    assert os.path.isdir(tmp_dir), 'Tmp dir must exists!'

    redis_dir = tempfile.mkdtemp(prefix='redis_')
    log(f'Redis dir: {redis_dir}')
    assert os.path.isdir(redis_dir), 'Redis dir must exists!'

    tgz = tarfile.open(REDIS_TGZ)
    tgz.extractall(path=redis_dir)

    pm = PortManager()
    redis_port = str(pm.get_port())

    cmd = [
        os.path.join(redis_dir, 'redis-server'),
        '--port',
        redis_port,
    ]
    log(f"Calling: {' '.join(cmd)}")
    server = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    set_env('REDIS_PID', str(server.pid))
    with open('redis_recipe.pid', 'w') as f:
        f.write(str(server.pid))

    deadline = time.time() + 5.0
    started = False
    while time.time() < deadline:
        try:
            redis.StrictRedis(port=int(redis_port)).ping()
        except redis.ConnectionError:
            time.sleep(0.1)
        else:
            started = True
            break
    if not started:
        raise Exception('Redis failed to start')

    set_env('REDIS_RECIPE_HOST', 'localhost')
    set_env('REDIS_RECIPE_PORT', redis_port)


def __is_running(pid):
    try:
        os.kill(pid, 0)
    except OSError as err:
        if err.errno == errno.ESRCH:
            return False
    return True


def stop_recipe_process_with_retries(process_pid: int, process_name: str):
    log(f'Stop {process_name}')
    try:
        os.kill(process_pid, 15)
    except Exception as e:
        logging.error(f'Failed to stop {process_name}: {e}')

    _SHUTDOWN_TIMEOUT = 10
    seconds = _SHUTDOWN_TIMEOUT
    while __is_running(process_pid) and seconds > 0:
        time.sleep(1)
    seconds -= 1

    if __is_running(process_pid):
        logging.error(f'{process_name} is still running after {seconds} seconds')
        os.kill(process_pid, 9)

    if __is_running(process_pid):
        logging.error(f'{process_name} failed to shutdown after kill 9!')


def stop(argv):
    if 'SKIP_CLEANUP_REDIS' in os.environ:
        return

    with open('redis_recipe.pid', 'r') as f:
        pid = int(f.read())

    stop_recipe_process_with_retries(process_pid=pid, process_name='redis')


if __name__ == '__main__':
    declare_recipe(start, stop)
