"""
Есть неявный доступ - унаследованный, неопределенный. Явные доступы бывают разные:
  * анонимный
  * как во всей вики
  * только авторам страницы
  * ограниченный (пользователями, группами и авторам)

"""
from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group as DjangoGroup
from wiki.legacy.choices import Choices

from wiki.notifications import signals as mail_signals
from wiki.pages import signals as page_signals
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.access.groups import user_django_groups
from wiki.pages.models import Access
from wiki.pages.tasks import UpdateClusterForIndexerTask
from wiki.utils.db import on_commit
from wiki.utils.idm import GroupNotFoundInIDM, get_group_roles


class NoChanges(Exception):
    """Никаких измений сделано не было"""

    pass


class NoUsersAndGroups(Exception):
    """ Чтобы поменять доступ, надо обязательно передать хоть одну группу или пользователя """

    pass


TYPES = Choices(
    ANONYMOUS='is_anonymous',  # используется во внешних вики
    COMMON='is_common',
    OWNER='is_owner',
    RESTRICTED='is_restricted',
    INHERITED='is_inherited',
)


def set_access_nopanic(
    page, access_type, author_of_changes, send_notification_signals=True, staff_models=None, groups=None
):
    """
    Обычно функции возвращают результаты не через raise, а через return
    NoUsersAndGroups при этом вполне ок ошибка
    Но пока нет ресурсов на рефакторинг acl, пусть будет враппер
    """
    try:
        set_access(page, access_type, author_of_changes, send_notification_signals, staff_models, groups)
        return True
    except NoChanges:
        return False


def set_access(page, access_type, author_of_changes, send_notification_signals=True, staff_models=None, groups=None):
    """ " установить права доступа к странице page
    @param page: wiki.pages.models.Page
    @param access_type: значение из TYPES
    @param author_of_changes users.User
    @param staff_models: список из intranet.models.Staff
    @param groups: список из intranet.models.Group (для внешних инстансов из django.contrib.auth.Group)

    @raises NoChanges
    @raises NoUsersAndGroups

    Департамент "Яндекс" будет отфильтрован из списка групп
    """
    assert isinstance(author_of_changes, get_user_model()), 'author_of_changes must be of User type, now - {0}'.format(
        type(author_of_changes)
    )

    # проверяем, что access_type in TYPES, будет брошено KeyError
    TYPES[access_type]  # noqa

    raw_access = get_bulk_raw_access([page])[page]
    access = interpret_raw_access(raw_access)
    removed_users, removed_groups, added_users, added_groups = [], [], [], []

    # унаследовать доступ - записей в Access быть не должно
    if access_type == TYPES.INHERITED:
        # возможно, ничего не поменяется
        if access['is_inherited']:
            raise NoChanges()

        for a in raw_access['list']:
            if a.page == page:
                a.delete()
        mail_signals.access_granted.send(
            sender=__file__, access=Access(page=page), author=author_of_changes, inherited=True, notify=False
        )
    # как во всей вики или только для авторов, или анонимный (одна запись в Access с соответсвующим флагом)
    elif access_type in (TYPES.COMMON, TYPES.OWNER, TYPES.ANONYMOUS):
        # возможно, ничего не поменяется
        if not access['is_inherited'] and (
            (access_type == TYPES.COMMON and access['is_common'])
            or (access_type == TYPES.OWNER and access['is_owner'])
            or (access_type == TYPES.ANONYMOUS and access['is_anonymous'])
        ):
            raise NoChanges()

        is_saved = False
        for a in raw_access['list']:
            if a.page == page:
                if not is_saved:
                    reset_access(a)
                    setattr(a, access_type, True)
                    a.save()
                    is_saved = True
                    if send_notification_signals:
                        mail_signals.access_granted.send(
                            sender=__file__, access=a, author=author_of_changes, notify=False
                        )
                else:
                    a.delete()
                    if send_notification_signals:
                        mail_signals.access_revoked.send(sender=__file__, access=a, author=author_of_changes)
        if not is_saved:
            a = Access(page=page)
            setattr(a, access_type, True)
            a.save()
            if send_notification_signals:
                mail_signals.access_granted.send(sender=__file__, access=a, author=author_of_changes, notify=False)

    # ограниченный доступ - одна и более записей в модели Access
    elif access_type == TYPES.RESTRICTED:
        staff_models = list(staff_models or [])
        groups = list(groups or [])

        if not (staff_models or groups):
            raise NoUsersAndGroups()

        # check if nothing has been changed
        if (
            not access['is_inherited']
            and access['is_restricted']
            and set(staff_models) == set(access['users'])
            and set(groups) == set(access['groups'])
        ):
            # с таким запросом в правах доступа менять нечего
            raise NoChanges()

        # пропустить пользователей и группы, которые не поменялись
        for a in raw_access['list']:
            if a.page != page:
                continue
            if a.staff and staff_models and a.staff in staff_models:
                staff_models.remove(a.staff)
            elif a.group and groups and a.group in groups:
                groups.remove(a.group)
            else:
                if a.staff:
                    removed_users.append(a.staff)
                elif a.group:
                    removed_groups.append(a.group)

                if send_notification_signals:
                    mail_signals.access_revoked.send(sender=__file__, access=a, author=author_of_changes)
                a.delete()

        # сохранить новых пользователей и новые группы
        if staff_models:
            added_users = staff_models
            for user in staff_models:
                a = Access(page=page, staff=user)
                a.save()
                mail_signals.access_granted.send(sender=__file__, access=a, author=author_of_changes, notify=False)
        if groups:
            added_groups = groups
            for group in groups:
                # здесь иногда проскакивает группа Яндекса, которая берется из кэша и может быть получена ранее
                # со slave инстанса базы, что ведет к ошибке вида:
                # Cannot assign "<Group: Яндекс>": instance is on database "default", value is on database "slave"
                # Потому группы сохраняем по id
                a = Access(page=page, group_id=group.id)
                a.save()
                if send_notification_signals:
                    mail_signals.access_granted.send(sender=__file__, access=a, author=author_of_changes, notify=False)

    # Изменение прав доступа может повлиять на права доступа подстраниц. Они должны попасть в индекс, поэтому им
    # всем нужно обновить modified_at_for_index. Такую операцию дорого проводить в реалтайме, делаем фоновую задачу.
    task = UpdateClusterForIndexerTask()
    on_commit(lambda: task.delay(page.id))

    old_access_type = (
        (access['is_restricted'] and TYPES.RESTRICTED)
        or (access['is_inherited'] and TYPES.INHERITED)
        or ((access['is_common'] or access['is_common_wiki']) and TYPES.COMMON)
        or (access['is_owner'] and TYPES.OWNER)
        or None
    )

    if send_notification_signals:
        mail_signals.page_access_settings_changed.send(
            sender=__file__,
            page=page,
            author_of_changes=author_of_changes,
            old_access_type=old_access_type,
            new_access_type=access_type,
            added_users=added_users,
            added_groups=added_groups,
            removed_users=removed_users,
            removed_groups=removed_groups,
        )

    page_signals.access_changed.send(sender=__file__, page_list=[page])
    return True


def reset_access(access):
    access.is_common = False
    access.is_owner = False
    access.is_anonymous = False
    access.staff = None
    access.group = None


def is_outstaff_manager(current_user):
    return any(
        group.name == settings.IDM_ROLE_OUTSTAFF_MANAGER_GROUP_NAME for group in user_django_groups(current_user)
    )


def get_outstaff_manager_idm_role_request(current_user):
    outstaff_manager_group_id = DjangoGroup.objects.get(name=settings.IDM_ROLE_OUTSTAFF_MANAGER_GROUP_NAME).id

    return (
        'https://idm.yandex-team.ru/system/{system}/roles#'
        'rf=1,rf-role={hash}#{login}@{system}/group-{group_id}'
        ';;;,rf-expanded={hash},sort-by=name'.format(
            system=settings.IDM_SYSTEM_NAME,
            hash='x',
            login=current_user.username,
            group_id=outstaff_manager_group_id,
        )
    )


def extract_outstaff(users):
    if not settings.IDM_ENABLED:
        return []

    outstaff_users = []
    for user in users:
        registered = any(
            group.name in (settings.IDM_ROLE_EMPLOYEE_GROUP_NAME, settings.IDM_ROLE_EXTERNAL_GROUP_NAME)
            for group in user_django_groups(user.user)
        )
        if not registered:
            outstaff_users.append(user)
    return outstaff_users


def extract_outstaff_and_deleted_groups(groups):
    """
    Операция выполняется за счет похода в IDM, поэтому оба в одной функции
    :param groups:
    :return:
    """
    if not settings.IDM_ENABLED:
        return [], []

    removed_groups, outstaff_groups = [], []

    for group in groups:
        if group.is_wiki or group.is_service or group.is_servicerole:
            # Считаем, что ABC-группы и вики-группы при добавлении в доступы к странице не требуют дополнительного
            # подтверждения от кого-либо с ролью OutstaffManager (WIKI-11502)
            registered = True
        else:
            try:
                registered = any(
                    role
                    in (settings.IDM_ROLE_EMPLOYEE_GROUP_NAME.lower(), settings.IDM_ROLE_EXTERNAL_GROUP_NAME.lower())
                    for role in get_group_roles(group)
                )
            except GroupNotFoundInIDM:
                removed_groups.append(group)
                continue

        if not registered:
            outstaff_groups.append(group)

    return outstaff_groups, removed_groups
