from typing import List, Optional, Dict
from enum import Enum
from pydantic import BaseModel


class CheckPermissionResult(Enum):
    ALLOW = 0
    DENY = 1
    NO_RESULT = 2


class ParentAclType(Enum):
    PAGE_ACL = '0'
    ROOT_ACL = '1'


class Action(Enum):
    READ = 'read'
    WRITE = 'write'
    MANAGE = 'manage'
    CHANGE_AUTHORS = 'change_authors'
    CHANGE_ACL = 'change_acl'
    DELETE = 'delete'


class AclListType(Enum):
    READONLY = 'r'
    READ_WRITE = 'rw'
    FULL_ACCESS = 'rwm'
    DENY = '!deny'


# список операций, допустимый для субъектов из соответствующего типа списка ACL
ALLOWED_ACTIONS_BY_ACL_LIST_TYPES: Dict[AclListType, List[Action]] = {
    AclListType.FULL_ACCESS: [Action.READ, Action.WRITE, Action.MANAGE],
    AclListType.READ_WRITE: [Action.READ, Action.WRITE],
    AclListType.READONLY: [Action.READ],
}


class AclMetaGroup(Enum):
    ALLSTAFF = 1
    SUPPORT = 2
    AUTHORS = 3
    SUPERUSERS = 4


class AclSource(Enum):
    WIKI = 'wiki'
    IDM = 'idm'
    CLOUD = 'cloud'


class AclSubjectType(Enum):
    GROUP = 'group'
    USER = 'user'
    META_GROUP = 'meta'


class AclSubject(BaseModel):
    subj_id: int
    subj_type: AclSubjectType

    def __eq__(self, other):
        return (
            isinstance(other, AclSubject)
            and self.subj_id_str == other.subj_id_str
            and self.subj_type == other.subj_type
        )

    def __lt__(self, other):
        return self.subj_id_str < other.subj_id_str

    def __hash__(self):
        return hash(self.subj_id_str) & hash(self.subj_type)

    @property
    def subj_id_str(self):
        # возвращает строковый id для пользователей и групп в формате "u2324" или "g11100" соответственно
        if self.subj_type == AclSubjectType.USER:
            return f'u{self.subj_id}'
        elif self.subj_type == AclSubjectType.GROUP:
            return f'g{self.subj_id}'
        else:
            return f'm{self.subj_id}'


class AclRecord(BaseModel):
    list_name: AclListType
    members: List[AclSubject]
    source: Optional[AclSource] = None

    def __eq__(self, other):
        return (
            isinstance(other, AclRecord)
            and self.list_name == other.list_name
            and self.source == other.source
            and set(self.members) == set(other.members)
        )

    def __hash__(self):
        return hash(self.list_name) & hash(tuple(sorted(self.members))) & hash(self.source)


class AclDiff(BaseModel):
    add: List[AclRecord] = []
    rm: List[AclRecord] = []

    def __eq__(self, other):
        return isinstance(other, AclDiff) and set(self.add) == set(other.add) and set(self.rm) == set(other.rm)

    def __hash__(self):
        return hash(tuple(sorted(self.add))) & hash(tuple(sorted(self.rm)))


class PageAcl(BaseModel):
    page_id: Optional[int] = None
    items: Dict[AclListType, List[AclSubject]]

    def __eq__(self, other):
        return (
            isinstance(other, PageAcl)
            and self.page_id == other.page_id
            and set(self.items.keys()) == set(other.items.keys())
            and all(set(other.items[k]) == set(v) for k, v in self.items.items())
        )
