import logging
from typing import List, Dict, Iterable, Union
from collections import defaultdict

from django.contrib.auth import get_user_model

from wiki.acl.consts import (
    AclDiff,
    Action,
    AclListType,
    AclSource,
    AclRecord,
    PageAcl,
    ALLOWED_ACTIONS_BY_ACL_LIST_TYPES,
    CheckPermissionResult,
    ParentAclType,
    AclSubject,
    AclSubjectType,
    AclMetaGroup,
)
from wiki.acl.models import Acl
from wiki.acl.utils import (
    get_effective_acl_from_cache,
    put_effective_acl_in_cache,
    is_user_employee,
    is_user_admin,
    is_user_page_owner,
)
from wiki.pages.models import Page


logger = logging.getLogger(__name__)

User = get_user_model()

ACL_QUERY = '''
WITH RECURSIVE acl(id, page_id, break_inheritance, acl, parent_acl_id, parent_acl_type) AS (
    SELECT id, page_id, break_inheritance, acl, parent_acl_id, parent_acl_type FROM acl_acl WHERE page_id = %s
  UNION ALL
    SELECT prnt.id, prnt.page_id, prnt.break_inheritance, prnt.acl, prnt.parent_acl_id, prnt.parent_acl_type
    FROM acl, acl_acl prnt
    WHERE prnt.id = acl.parent_acl_id and acl.break_inheritance = False
  )
SELECT * FROM acl;'''


def get_effective_acl(page_id: int) -> PageAcl:

    # проверим кеш
    eff_page_acl = get_effective_acl_from_cache(page_id)
    if eff_page_acl:
        return eff_page_acl

    # эффективный acl из источником AclSource.WIKI. Содержит правила, которые можно удалять в потомках.
    effective_acl_from_wiki: Dict[AclListType, set[AclSubject]] = defaultdict(set)

    # acl родительской страницы. Содержит правила, которые нельзя удалять в потомках.
    effective_acl: Dict[AclListType, set[AclSubject]] = defaultdict(set)

    query_result = Acl.objects.raw(ACL_QUERY, [page_id])
    # Применяем acl от родительского списка до последнего дочернего
    for db_acl in reversed(list(query_result)):
        # Если наследование происходит от ROOT_ACL, то неявным корневым правилом будет "rwm": "%allstaff%"
        if db_acl.parent_acl_type == ParentAclType.ROOT_ACL and not db_acl.break_inheritance:
            for item in get_root_acl().add:
                effective_acl_from_wiki[item.list_name].update(item.members)

        if not db_acl.acl:
            continue

        acl = AclDiff.parse_raw(db_acl.acl)
        for item in acl.add:
            if item.source and item.source != AclSource.WIKI:
                effective_acl[item.list_name].update(item.members)
            else:
                effective_acl_from_wiki[item.list_name].update(item.members)

        for item in acl.rm:
            effective_acl_from_wiki[item.list_name].difference_update(item.members)

    for k, v in effective_acl_from_wiki.items():
        effective_acl[k].update(v)

    effective_acl = {k: list(v) for k, v in effective_acl.items()}
    eff_page_acl = PageAcl(page_id=page_id, items=effective_acl)

    # положим свежее значение в кеш
    put_effective_acl_in_cache(page_id, eff_page_acl)

    return eff_page_acl


def _user_or_group_in_members(acl_user: AclSubject, acl_groups: List[AclSubject], members: List[AclSubject]):
    return members and (acl_user in members or not set(acl_groups).isdisjoint(set(members)))


def _check_permission(
    acl_user: AclSubject, acl_groups: List[AclSubject], acl: PageAcl, action: Action
) -> CheckPermissionResult:
    if _user_or_group_in_members(acl_user, acl_groups, acl.items.get(AclListType.DENY, {})):
        return CheckPermissionResult.DENY

    for acl_list_type, action_list in ALLOWED_ACTIONS_BY_ACL_LIST_TYPES.items():
        if action in action_list and _user_or_group_in_members(acl_user, acl_groups, acl.items.get(acl_list_type, [])):
            return CheckPermissionResult.ALLOW

    return CheckPermissionResult.NO_RESULT


def _make_subj_list(ids: Iterable[Union[int, str]], subj_type) -> List[AclSubject]:
    return [AclSubject(subj_id=id, subj_type=subj_type) for id in ids]


def check_permission(page: Page, user: User, action: Action) -> bool:
    # если пользователь входит в мета группу %authors% или %superuser%, то доступ разрешен для любой операции
    if is_user_admin(user) or is_user_page_owner(user, page):
        return True

    user_groups = _make_subj_list(user.get_all_group_ids(), AclSubjectType.GROUP)
    if is_user_employee(user):
        user_groups.extend(_make_subj_list([AclMetaGroup.ALLSTAFF.value], AclSubjectType.META_GROUP))

    result = _check_permission(
        user.get_acl_subject(),
        user_groups,
        get_effective_acl(page.id),
        action,
    )
    return result == CheckPermissionResult.ALLOW


def get_root_acl() -> AclDiff:
    return AclDiff(
        add=[
            AclRecord(
                list_name=AclListType.FULL_ACCESS,
                members=_make_subj_list([AclMetaGroup.ALLSTAFF.value], AclSubjectType.META_GROUP),
            )
        ]
    )
