import logging
from typing import List, Callable, Union

from django.contrib.auth import get_user_model

from wiki.acl.consts import (
    AclDiff,
    AclListType,
    AclRecord,
    ParentAclType,
    AclSubjectType,
    AclMetaGroup,
)
from wiki.acl.errors import AclNotFoundException, UnknownPageAccessType
from wiki.acl.logic.calculation import _make_subj_list, get_root_acl
from wiki.acl.models import Acl
from wiki.acl.utils import invalidate_cache
from wiki.pages.models import Page
from wiki.pages.logic import access as access_logic


logger = logging.getLogger(__name__)

User = get_user_model()


def _get_children_acl(parent_acl_id: int) -> List[Acl]:
    return list(Acl.objects.filter(parent_acl_id=parent_acl_id))


def _update_acl_attrs_for_acl_list(acl_list: List[Acl], parent_acl: Union[Acl, None], parent_acl_type: ParentAclType):
    for acl in acl_list:
        _update_acl_attrs(acl, parent_acl, parent_acl_type)


def _update_acl_attrs(acl: Acl, parent_acl: Union[Acl, None], parent_acl_type: ParentAclType):
    if acl:
        acl.parent_acl = parent_acl
        acl.parent_acl_type = parent_acl_type
        acl.save()


def _get_all_parent_and_children_page_ids(page: Page) -> List[int]:
    result = list(page.ancestors.values_list('id', flat=True)) + list(page.descendants.values_list('id', flat=True))
    return result


def _get_page_acl(page: Page):
    try:
        return Acl.objects.get(page=page)
    except Acl.DoesNotExist:
        logger.error(f'Can\'t found Acl for page id={page.id} supertag={page.supertag}')
        raise AclNotFoundException()


def _insert_acl_to_new_place_in_chain(acl: Acl):
    parent_pages = list(acl.page.ancestors)
    children_page_ids = list(acl.page.descendants.values_list('id', flat=True))
    if parent_pages:
        parent_page = parent_pages[-1]
        parent_acl = _get_page_acl(parent_page)
        _update_acl_attrs(acl, parent_acl=parent_acl, parent_acl_type=ParentAclType.PAGE_ACL)
        children_acl = list(Acl.objects.filter(page_id__in=children_page_ids, parent_acl=parent_acl))
        _update_acl_attrs_for_acl_list(children_acl, parent_acl=acl, parent_acl_type=ParentAclType.PAGE_ACL)
    else:
        _update_acl_attrs(acl, parent_acl=None, parent_acl_type=ParentAclType.ROOT_ACL)
        children_acl = list(Acl.objects.filter(page_id__in=children_page_ids, parent_acl_type=ParentAclType.ROOT_ACL))
        _update_acl_attrs_for_acl_list(children_acl, parent_acl=acl, parent_acl_type=ParentAclType.PAGE_ACL)


def update_acl_after_page_deletion(page: Page):
    """
    Обновить цепочку ACL объектов, связанных с переданной страницы после операции удаления страницы.
    """
    acl = _get_page_acl(page)
    if acl.parent_acl:
        _update_acl_attrs_for_acl_list(
            _get_children_acl(acl.id), parent_acl=acl.parent_acl, parent_acl_type=ParentAclType.PAGE_ACL
        )
        _update_acl_attrs(acl, parent_acl=None, parent_acl_type=acl.parent_acl_type)
    else:
        _update_acl_attrs_for_acl_list(
            _get_children_acl(acl.id), parent_acl=None, parent_acl_type=ParentAclType.ROOT_ACL
        )

    invalidate_cache(_get_all_parent_and_children_page_ids(page))


def update_acl_after_page_restore(page: Page):
    """
    Обновить цепочку ACL для переданной страницы после операции восстановления удаленной страницы.
    """
    try:
        acl = _get_page_acl(page)
    except AclNotFoundException:
        acl = _create_acl(page)

    _insert_acl_to_new_place_in_chain(acl)


def update_acl_after_page_movement(page_redirect: Page):
    """
    Обновить после операции перемещения страницы цепочку ACL для переданной страницы-редиректа,
    а также для связанной с ней страницы, на которую указывает редирект.
    """
    moved_page = page_redirect.redirects_to
    moved_page_acl = _get_page_acl(moved_page)
    redirect_page_acl = _get_page_acl(moved_page).copy()
    redirect_page_acl.page = page_redirect
    redirect_page_acl.save()
    _insert_acl_to_new_place_in_chain(redirect_page_acl)
    _insert_acl_to_new_place_in_chain(moved_page_acl)


def update_acl_after_page_creation(page: Page):
    """
    Создать новый Acl для новой страницы и обновить цепочку ACL
    """
    _create_acl(page)


def _create_acl(page: Page) -> Acl:
    acl = _get_new_acl(page, _get_new_acl)
    acl.save()
    return acl


def _get_or_create_acl(page, create_acl_func: Callable[[Page], Acl]):
    try:
        return Acl.objects.get(page=page)
    except Acl.DoesNotExist:
        return create_acl_func(page)


def _get_new_acl(page: Page, create_acl_func: Callable[[Page], Acl]) -> Acl:
    parent_pages = list(page.ancestors)
    if len(parent_pages) > 0:
        parent_acl_type = ParentAclType.PAGE_ACL
        parent_acl = _get_or_create_acl(parent_pages[-1], create_acl_func)
    else:
        parent_acl_type = ParentAclType.ROOT_ACL
        parent_acl = None

    return Acl(
        page=page, break_inheritance=False, parent_acl_type=parent_acl_type, parent_acl=parent_acl, acl=AclDiff().json()
    )


def convert_page_access_to_new_acl(page: Page) -> Acl:
    """
    Преобразовать для заданной страницы старые настройки прав в новый ACL объект и сохранить в БД.
    В процессе преобразования необходимо рекурсивно создать и ACL объекты для родительских страниц, чтобы обеспечить
    иерархическую цепочку ACL.
    """
    try:
        return _get_page_acl(page)
    except AclNotFoundException:
        pass

    acl = _get_new_acl(page, convert_page_access_to_new_acl)
    page_access = access_logic.get_access(page)

    acl.break_inheritance = page_access.type in ['owner', 'restricted', 'common']

    acl_diff = AclDiff()
    if page_access.is_common:
        acl_diff = get_root_acl()
    elif page_access.is_owner:
        acl_diff.add = [
            AclRecord(
                list_name=AclListType.FULL_ACCESS,
                members=_make_subj_list([AclMetaGroup.AUTHORS.value], AclSubjectType.META_GROUP),
            )
        ]
    elif page_access.is_restricted:
        users = [user.get_acl_subject() for user in page_access.restrictions['users']]
        groups = [group.get_acl_subject() for group in page_access.restrictions['groups']]
        acl_diff.add = [
            AclRecord(
                list_name=AclListType.FULL_ACCESS,
                members=users + groups,
            )
        ]
    elif page_access.is_inherited:
        # оставляем дефолтные значения атрибутов, определенные выше по коду
        assert acl.parent_acl is not None
    else:
        raise UnknownPageAccessType()

    acl.acl = acl_diff.json()
    acl.save()

    return acl
