import logging
from typing import List, Union

from django.core.cache import caches
from django.conf import settings
from django.contrib.auth import get_user_model

from wiki.acl.consts import PageAcl
from wiki.pages.models import Page
from wiki.sync.connect.org_ctx import get_org
from wiki.users.models import Group
from wiki.utils.per_request_memoizer import memoize_for_this_request


IS_INTRANET = getattr(settings, 'IS_INTRANET', False)

CACHE_EFFECTIVE_ACL = caches['effective_acl']
CACHE_EFFECTIVE_ACL_TIMEOUT = 60 * 60  # in sec

User = get_user_model()

logger = logging.getLogger(__name__)


def get_effective_acl_from_cache(page_id: int) -> Union[PageAcl, None]:
    return CACHE_EFFECTIVE_ACL.get(page_id)


def put_effective_acl_in_cache(page_id: int, acl: PageAcl):
    CACHE_EFFECTIVE_ACL.set(page_id, acl, CACHE_EFFECTIVE_ACL_TIMEOUT)


def invalidate_cache(page_ids: List[int]):
    CACHE_EFFECTIVE_ACL.delete_many(keys=page_ids)


def is_user_page_owner(user: User, page: Page) -> bool:
    return page.get_authors().filter(staff=user.staff).exists()


if IS_INTRANET:

    def is_user_employee(user: User) -> bool:
        """
        Проверить входит ли пользователь в группу Employee в IDM.
        """
        return is_user_member_of_group(user, settings.IDM_ROLE_EMPLOYEE_GROUP_NAME)

    def is_user_admin(user: User):
        """
        Проверить входит ли пользователь в группу Admins в IDM.
        """
        return is_user_member_of_group(user, settings.IDM_ROLE_ADMINS_GROUP_NAME)

    @memoize_for_this_request
    def is_user_member_of_group(user, group_name):
        return user.groups.filter(name=group_name).exists()


else:

    @memoize_for_this_request
    def is_user_employee(user: User) -> bool:
        return user.is_authenticated

    @memoize_for_this_request
    def is_user_admin(user: User) -> bool:
        """
        Поверяет, состоит ли пользователь в группе администраторов организации
        """
        if user.is_authenticated:
            return Group.objects.filter(user=user, group_dir_type__iexact='organization_admin', org=get_org()).exists()
        return False
