import logging

from django.http import HttpResponse

from wiki.legacy.formatters.wf_config import config

logger = logging.getLogger('wiki.actions')

# Cache for actions' classes
_action_classes = {}
_missing_action_names = set()


class UnknownActionError(Exception):
    pass


class NotDynamicAction(Exception):
    pass


def get_action_class(action_name):
    if action_name in _missing_action_names:
        raise UnknownActionError("Action '%s' module is missing (called again)" % action_name)
    if action_name not in _action_classes:
        class_name = action_name.capitalize()
        module = None
        for package in config.actions_packages:
            module_name = package + '.' + action_name
            try:
                module = __import__(module_name, {}, {}, [str(class_name)])
                break
            except ImportError as e:
                if str(e).find("No module named '{0}'".format(module_name)):
                    raise
                logger.debug("No action '%s' module in '%s' package. Import result is '%s'" % (action_name, package, e))
                pass
        if module is None:
            logger.debug(
                "Couldn't find action '%s' module. Configured packages are '%s'" % (
                    action_name, config.actions_packages
                )
            )
            _missing_action_names.add(action_name)
            raise UnknownActionError("Action '%s' module is missing" % action_name)
        _action_classes[action_name] = getattr(module, class_name)
    return _action_classes[action_name]


def call_action_with_ctx(name, params, request, **ctx):
    """
    Helper function for calling actions from python code.
    Returns string with HTML or WOM object.
    """
    action = get_action_class(name, config)(params, request, **ctx)
    return action.render()


def call_action(name, params, request, page_path):
    """
    Вызвать метод render у вики-экшена.

    @type name: str
    @type params: dict
    @type request: HttpRequest
    @type page_path: str
    @param name: имя экшена
    @param params: параметры экшена
    @param page_path: тег страницы, на которой вызван экшен
    @rtype: str
    """
    from wiki.users.logic.settings import get_user_setting

    return call_action_with_ctx(
        name,
        params,
        request,
        page_path=page_path,
        code_theme=get_user_setting(request.user, 'code_theme'),
    )


def handle_action(name, **kwargs):
    """
    Helper function for handling POST request by a particular action.
    Handle_action and call_action have the same signature.
    Returns string with HTML, WOM object or HttpResponse object.
    """
    action = get_action_class(name)(**kwargs)
    if not hasattr(action, 'handle'):
        raise NotDynamicAction('Action "%s" is not dynamic' % name)

    r = action.handle()
    if isinstance(r, HttpResponse):
        return r
    return action.render()
