import json
import logging

from django.http import Http404
from django.utils.translation import ugettext

from wiki.actions.classes.base_action_deprecated import StopError, WikiActionWithPOST
from wiki.org import get_org
from wiki.pages.access import ACCESS_DENIED, ACCESS_NONEXISTENT, get_bulk_access_status
from wiki.pages.models import Page, PageLink
from wiki.utils.supertag import tag_to_supertag

logger = logging.getLogger(__name__)


class Backlinks(WikiActionWithPOST):
    """
    Экшн {{backlinks}} -- отображет в виде списка страницы, ссылающиеся на данную.

    Параметры:
        {for} - tag страницы, для которой строится список, если не указан, берется та, с которой экшн вызван.
                Также по историческим причинам любой не упомянутый далее параметр, переданный этому экшну,
                считается параметром for.
        {nomark} - флаг того, что не надо обрамлять результат в тэги <fieldset/>
    """

    default_params = {}

    @property
    def is_json_rendering(self):
        return self.params.get('render_json') is not None

    @is_json_rendering.setter
    def is_json_rendering(self, value):
        # игнорируем setter, поскольку значение зависит только от GET параметра 'render_json'
        pass

    def render_json(self):
        try:
            data = self.get_context_data()['data']
        except StopError:
            # Не удалось найти страницу в переданным tag
            raise Http404()

        return json.dumps(data)

    def get_context_data(self):
        """
        Вернуть контекст для экшна {{backlinks}}
        """
        self.adapt_params()

        # получим данные по всем страницам, ссылающимся на нужную
        backlinks_list = self.get_backlinks_list(root_page=self.params['root'])

        # уберем недоступные текущему пользователю ссылки
        if self.request.from_yandex_server:
            backlinks_list = []
        else:
            backlinks_list = self.filter_accessible(backlinks_list, self.request.user)

        return {'data': backlinks_list, 'nomark': self.params.get('nomark', False)}

    def adapt_params(self):
        """
        Очистить и привести к нужному виду параметры экшна
        """
        # заполняем дефолтами нужные поля
        [self.params.setdefault(key, value) for key, value in list(self.default_params.items())]

        # получаем страницу, для которой вызван экшн
        self.params['root'] = self.get_root_page()

        # nomark таки тоже должен работать, если указан
        if 'nomark' in self.params:
            self.params['nomark'] = True

    def get_root_page_tag_from_params(self):
        """
        Вернуть ссылку на страницу, указанную в вики тексте действия
        или None, если страница явно не указана в параметрах.
        """
        page_tag = None
        if self.is_param('for'):
            page_tag = self.params['for']
        elif len(self.ordered_params) and (
            not self.is_param('nomark') or self.params.get('nomark') != self.ordered_params[0]
        ):
            page_tag = self.ordered_params[0]

        return page_tag

    def get_root_page(self):
        """
        Вернуть страницу, для которой выполняется экшн.
        Тэг страницы берется из параметра 'for', если такой указан, если не указан,
        то тэг ищется просто как первый именованный параметр в экшне.
        Если и там не нашли - берется страница, на которой экшн вызван.

        @rtype: Page
        """
        page_tag = self.get_root_page_tag_from_params()

        try:
            if page_tag:
                root_page = Page.active.get(supertag=tag_to_supertag(page_tag), org=get_org())
            else:
                root_page = self.page
        except Exception:
            self.add_error_and_stop(
                # Translators: Ошибка -- экшен backlinks не получил адрес корневой страницы
                ugettext('actions.Backlinks:NoPageAddress')
            )
        else:
            return root_page

    @staticmethod
    def get_backlinks_list(root_page):
        """
        Вернуть данных по активным страницам, ссылающихся на переданную

        @type root_page: Page
        @rtype: list
        """
        result = []

        # все id страниц, ссылающихся на переданную
        linked_pages_ids = PageLink.objects.filter(to_page=root_page).values_list('from_page_id', flat=True)

        # возвращаем лишь активные страницы
        linked_pages = Page.active.filter(id__in=set(linked_pages_ids))

        for page in linked_pages:
            # готовим данные для шаблона экшна
            result.append(
                {
                    'name': '{page_name}'.format(page_name=page.tag.replace('/', '.')),
                    'link': '{page_link}#{root_supertag}'.format(page_link=page.tag, root_supertag=root_page.supertag),
                    'label': page.tag,
                }
            )

        return result

    @staticmethod
    def filter_accessible(backlinks_list, user):
        """
        Вернуть фильтрованный от недоступных пользователю страниц список данных

        @type backlinks_list: list
        @param backlinks_list: список данных для экшна backlinks
        @type user: User
        @param user: Django пользователь, который производит запрос

        @rtype: list
        @return: фильтрованный от недоступных страниц список данных для экшна backlinks
        """
        # тэги проверяемых страниц
        pages_tags = set([entry['label'] for entry in backlinks_list])

        # получаем данные по доступу пользователя к страницам в терминах ACCESS_COMMON, .. ACCESS_DENIED
        access_status = get_bulk_access_status(pages_tags, user)

        # фильтруем исходные данные
        return [
            entry
            for entry in backlinks_list
            if access_status[entry['label']] not in (ACCESS_DENIED, ACCESS_NONEXISTENT)
        ]
