import logging

from wiki.api_core.errors.rest_api_error import RestApiError
from wiki.api_frontend.logic.errors import simplify_grid_row_errors
from wiki.utils.errors import InputValidationError
from wiki.utils.link_to_normalized_supertag import link_to_normalized_supertag

log = logging.getLogger(__name__)


class ActionError(RestApiError):
    error_code = 'ACTION_PRODUCED_ERROR'
    debug_message = 'Action produced error'


class ParamsWrapper:
    """Обертка вокруг словаря и списка с параметрами действия/форматера.
    Предоставляет методы для получения значения параметра по имени и по индексу.
    """

    __WITHOUT_DEFAULT = object()

    def __init__(self, dict, list=None):
        self.dict = dict
        self.list = list

    def value_by_name(self, name, default=__WITHOUT_DEFAULT):
        """Возвращает значение параметра с именем name.
        Аргумент default задаёт значение на случай, если параметр с указанным именем отсутствует.
        Выбрасывает KeyError, если default не задан и параметр с указанным именем отсутствует.
        (default=None - допустимое значение по-умолчанию.)
        """
        return self.dict[name] if default is ParamsWrapper.__WITHOUT_DEFAULT else self.dict.get(name, default)

    def bool_value_by_name(self, name, default=__WITHOUT_DEFAULT):
        """Возвращает bool-значение параметра с именем name.
        Аргумент default задаёт значение на случай, если параметр с указанным именем отсутствует.
        (default обязан быть типа bool)
        Выбрасывает KeyError, если default не задан и параметр с указанным именем отсутствует.
        Как False трактуются лишь 2 значения параметра:
            * пустая строка ("")
            * строка содержащая один символ 0 ("0")
        (Числовой - исключён, потому что PARAM-узлы WOM-модели содержат лишь строки)
        """
        if default is ParamsWrapper.__WITHOUT_DEFAULT:
            return self.dict[name]
        else:
            assert isinstance(default, bool)
            v = self.dict.get(name, default)
            return v if v == default else not (v == '' or v == '0')

    def value_at(self, idx):
        """Возвращает параметр по 0-based индексу idx.
        Возвращает тапл из 2-х элементов:
            * (имени, значения) – для именнованного PARAM-узла
            * (None, значения) – для неименнованного PARAM-узла
        Выбрасывает IndexError, если idx некорректен.
        """
        return self.list[idx]


class WikiBaseAction(object):
    """
    Базовый экшен для nodejs-верстки.

    Умеет отдавать форматтеру на операцию view словарь.
    Это JSON-результат его работы.
    Форматтер на операцию view вставляет результат в специальную конструкцию
    внутри bemjson.
    """

    page = None
    user = None
    request = None
    params = None

    def __init__(self, params, request, **kwargs):
        """
        @type params: ParamsWrapper
        """
        self.page = request.page
        self.user = request.user
        self.params = params
        self.request = request

    def encode_params(self, params):
        """
        Закодировать ParamsWrapper для фронтэнда в виде маппинг ключ-значение.

        Пары ключ-значение из экшенов становятся так же парами ключ-значение.
        Упорядоченные параметры без значения становятся ключ-пустая строка. Т.е.
        свойство упорядоченности на фронтэнд не передается, теряется.

        Все ordered_params, которые были при указании экшена, необходимо так
        преобразовывать в методе encode_params, чтобы фронтэнду было просто работать
        с ними.

        Пример 1:

        если {{grid}} принимает page двумя способами
          * {{grid page='tag'}}
          * {{grid tag}}
        То в результате работы encode_params нужно всегда передавать 'tag' в
        параметре "page". Т.е. нужно переопределить encode_params в экшене
        grid, если вы используете упорядоченный параметр как значение.

        Пример 2:

        {{include page="tag" nomark notitle}}
        если экшен include принимает параметр nomark как ordered, то можно ничего не делать,
        просто учитывать что в url_with_data будет вот так:
          url_with_data="...?...&nomark=&notitle="
        То есть параметры будут переданы один за другим. Тут упорядоченные параметры
        используются как имена.

        @type params: ParamsWrapper
        @rtype: dict
        """
        result = params.dict or {}
        if params.list:
            # param_name параметров, которые указываются позиционно без значений
            # кодируются в None
            for param_name, value in params.list:
                if param_name is None:
                    result[value] = ''
        return result

    @staticmethod
    def ordered_params(params):
        """
        Вернуть ordered_params.

        @type params: ParamsWrapper
        @rtype generator
        """
        return (kv[1] for kv in (params.list or []) if kv[0] is None)

    @property
    def default_params(self):
        """
        Значения по умолчанию для тех параметров, которые не были указаны.
        """
        return {}

    def form_class_to_validate_params(self):
        """
        Вернуть форму для валидации именованных параметров.

        Вызывается уже после того, как параметры были закодированы в dict через
        encode_params. Если метод не реализовать, то считается что валидация не
        требуется.

        Помните, что для валидации булевых значений нужно использовать
        classes.params_validator.BooleanFromTextField

        Можно
          * не реализовывать вовсе
          * реализовывать только для тех параметров, которые нужно провалидировать.

        @rtype: django.forms.Form
        """
        raise NotImplementedError()

    def render(self):
        """
        Метод реализует интерфейс экшена с точки зрения форматтера.
        """
        try:
            form_class = self.form_class_to_validate_params()
        except NotImplementedError:
            form_class = None

        encoded_params = self.encode_params(self.params)

        # create defaults
        for param, default_value in self.default_params.items():
            if param not in encoded_params:
                encoded_params[param] = default_value

        if form_class:
            form = form_class(encoded_params)
            if not form.is_valid():
                # нужно вывести ошибку, объединив все возможные ошибки в одну строку
                raise InputValidationError(', '.join(simplify_grid_row_errors(form.errors)))
            cleaned_params = {k: v for k, v in form.cleaned_data.items() if k in list(encoded_params.keys())}
            encoded_params.update(cleaned_params)

        return self.json_for_formatter_view(page=self.page, user=self.user, params=encoded_params)

    def json_for_formatter_view(self, page, user, params):
        """
        Вернуть JSON для операции VIEW форматтера.

        @type page: Page
        @param page: может быть None.
        @type user: User
        @type params: dict
        @rtype: dict
        """
        raise NotImplementedError()

    def error_happened(self, message):
        """
        Произвести ошибку. Будет означать, что нужно отобразить ошибку,
        как если бы она произошла в REST-API.

        Представление ошибки см в автодокументации, в разделе
        "При запросе произошла ошибка".

        @type message: basestring
        """
        raise InputValidationError(message)

    def normalized_supertag(self, tag, source_page=None):
        if source_page:
            supertag = source_page.supertag
        else:
            supertag = self.page.supertag

        return link_to_normalized_supertag(tag, supertag)

    def __repr__(self):
        return 'Action {0} of nodejs frontend'.format(self.__class__.__name__)
