from urllib.parse import urlparse

from django.conf import settings
from django.contrib import messages
from django.http import Http404, HttpResponse, HttpResponseRedirect
from django.template import loader
from django.utils.encoding import smart_bytes
from django.utils.http import urlencode
from django.utils.translation import gettext
from django.views.generic.base import View
from ujson import loads
from ylog.context import log_context

from wiki.actions.utils import END_DYNAMIC_TAG, START_DYNAMIC_TAG
from wiki.org import get_org, org_ctx
from wiki.pages.access.groups import user_django_groups
from wiki.pages.models import Page
from wiki.utils.errors import InputValidationError
from wiki.utils.backports.mds_compat import APIError
from wiki.utils.path import canonize, resolve
from wiki.utils.request_logging.context import extract_normalized_route
from wiki.utils.supertag import translit


class StopError(Exception):
    """
    Ошибка для метода add_error_and_stop
    """

    pass


class BaseAction(object):
    """
    Интерфейс экшена из вики-форматтера.
    """

    # Contains a dict which will specify default values for action's params.
    default_params = None

    def __init__(self, params=None, request=None, **kwargs):
        self.name = self.__class__.__name__.lower()

        self.params = {'errors': []}
        if self.default_params:
            self.params.update(self.default_params)
        self.ordered_params = []
        if params and hasattr(params, 'dict'):
            self._parse_wf_params(params)
        elif params:
            self.params.update(params)
        if request:
            self.request = request

        for key, value in kwargs.items():
            setattr(self, key, value)

    def _parse_wf_params(self, params):
        if params.dict:
            self.params.update(params.dict)
        if params.list:
            self.ordered_params.extend(p[1] for p in params.list)

    def render(self):
        """
        Результатом работы экшена может быть WOM-узел или строка.
        """
        raise NotImplementedError()


class OldDjangoWikiBaseAction(BaseAction):
    """
    Базовый экшен вики.

    Умеет рендерить шаблон, находить его по имени класса.
    Умеет рендерить ошибку на месте себя.

    DEPRECATED. После закрытия старой верстки надо удалить.
    """

    # Specifies template's name for rendering.
    # If is None, actions/<classname>.html will be used as template.
    template_name = None

    # If True self.params will be base of context for template rendering.
    params_to_context = True

    # If True and errors have been occured, _errors.html will be rendered instead
    # of action's template.
    display_errors = True

    # Number of action on page among actions of one class.
    # Set automatically by WF or parsing HTTP params.
    number = None

    # Mode ID of WF which call the action, e.g. 'view' or 'submit'.
    # Set by new WF only.
    view_mode_id = None

    # Normalized supertag of page where action has been called.
    # Set by new WF only. Used in self.page property.
    page_path = None

    default_params = {
        'settings': settings,
    }

    def __init__(self, params=None, *args, **kwargs):
        self.http_status_code = 200
        super(OldDjangoWikiBaseAction, self).__init__(params, *args, **kwargs)

        # отдельный список для предупреждений на странице
        self.params['warnings'] = []
        if params and isinstance(params, dict) and isinstance(params.get('page'), str):
            self.page_path = params.get('page')

    def is_param(self, name):
        value = self.params.get(name, False)
        if value and str(value).lower() not in ('0', 'false', 'none') or name in self.ordered_params:
            return True
        return False

    def add_warning(self, warning):
        """
        добавляет warning-сообщение, которое будет выводиться
        вместе (как правило сверху)
        с html-ом с помощью шаблона actions/_warnings.html для тех экшнов,
        у которых он подключен

        @type warning: unicode
        """
        self.params['warnings'].append(warning)

    def add_error(self, error):
        """
        добавляет error-сообщение, которое будет выводиться
        вместо html экшна с помощью шаблона actions/_errors.html

        @type error: unicode
        @return: False
        """
        self.params['errors'].append(error)
        return False

    def add_error_and_stop(self, error):
        """
        добавляет error-сообщение, которое будет выводиться
        вместо html экшна с помощью шаблона actions/_errors.html
        и кидает StopError

        @type error: unicode
        @raise: StopError
        """
        self.add_error(error)
        raise StopError()

    def render(self):
        """
        Renders action's template and returns the result of the action.
        Main method in actions API.
        """
        ctx = self._get_ctx()
        if ctx.get('errors') and self.display_errors:
            return self._render_errors()
        self._update_ctx(ctx)

        if not self.template_name:
            tpl_name = 'actions/%s.html' % self.name
        else:
            tpl_name = self.template_name

        return loader.render_to_string(tpl_name, ctx)

    def _render_errors(self, action_body=False):
        tpl_name = 'actions/_errors.html'
        if 'action_body' not in self.params:
            self.params['action_body'] = action_body
        return loader.render_to_string(tpl_name, self.params)

    def _update_ctx(self, ctx):
        ctx['action_name'] = self.name
        ctx['action_number'] = self.number

    def _get_ctx(self):
        """
        Returns calculated context for template's rendering according
        to self.params_to_context and the result of self.set_context_data
        or self.get_context_data methods.
        """
        if self.params_to_context:
            ctx = self.params
        else:
            ctx = {}
        try:
            ctx = self._update_by_subclass(ctx)
        except StopError:
            pass
        return ctx

    def _update_by_subclass(self, ctx):
        if hasattr(self, 'set_context_data'):
            self.set_context_data()
        elif hasattr(self, 'get_context_data'):
            r = self.get_context_data()
            if r:
                if self.params_to_context:
                    ctx.update(r)
                else:
                    ctx = r
        return ctx


class OldDjangoWikiAction(OldDjangoWikiBaseAction, View):
    """
    Экшен для Вики. Умеет отвечать двумя способами на GET-запрос:

      * при вызове операции view форматтера
      * при вызове по прямому урлу (настроено в urls.py)

    DEPRECATED. После закрытия старой верстки надо удалить.
    """

    http_method_names = ['get']
    http_content_type = 'text/html'
    _is_http_params_parsed = False

    def dispatch(self, request, *args, **kwargs):

        # здесь происходит хак только для http-экшенов - tag получаем из urls.py джанги.
        self._page = request.page
        route = extract_normalized_route(request)
        endpoint = self.__class__.__name__

        with org_ctx(request.org), log_context(route=route, endpoint=endpoint):
            return super(OldDjangoWikiAction, self).dispatch(request, *args, **kwargs)

    def get(self, *args, **kwargs):
        self._parse_http_params()
        r = self.render()
        return HttpResponse(r, self.http_content_type)

    def check_user_access(self):
        # По требованию СИБ запрещаем в интранете внешним пользователям доступ к внешним страницам и файлам
        # через такие действия как include и csvfile.
        # см. https://st.yandex-team.ru/WIKI-11459
        if settings.IS_INTRANET and all(
            group.name != settings.IDM_ROLE_EMPLOYEE_GROUP_NAME for group in user_django_groups(self.request.user)
        ):
            raise InputValidationError(
                'Content of the included page or file should be displayed here. '
                'But you have no access to the included content.'
            )

    def _parse_http_params(self):
        """
        Parses params from HTTP request.
        """
        # Stop if already parsed
        if self._is_http_params_parsed:
            return

        # Parse parameters without values, e.g {{action xxx yyy}}
        ordered_params = self.request.POST.get('__no_values_params__', self.request.GET.get('__no_values_params__'))
        if ordered_params:
            self.ordered_params = loads(smart_bytes(ordered_params))

        idxs = []
        data = self.request.GET.copy()
        data.update(self.request.POST)
        for name, value in data.items():
            if name.isdigit():
                idxs.append(int(name))
            else:
                self.params[name] = value
                if name.lower() != name:
                    self.params[name.lower()] = value
        if idxs:
            idxs.sort()
            for i in idxs:
                self.ordered_params.append(self.request.POST.get(str(i), self.request.GET.get(str(i))))
        self._is_http_params_parsed = True
        if not getattr(self, 'number') and self.params.get('__count__'):
            try:
                self.number = int(self.params.get('__count__'))
            except ValueError:
                pass
        if hasattr(self, 'message_type'):
            self.message_type = self.params.get('__message_type__', self.message_type)

    @staticmethod
    def _get_page(tag):
        """
        Низкоуровневый геттер страницы, работает с тегами

        @type tag: str
        @rtype: Page | None
        """

        if tag is None:
            return

        # Вырезаем имя хэндлера из тега, если есть
        tag_chunks = tag.split('/')
        if tag_chunks[-1].startswith('.'):
            tag = '/'.join(tag_chunks[:-1])

        # Есть старые страницы с точкой на конце тега
        supertag = translit(tag.rstrip('.')) or settings.MAIN_PAGE

        try:
            return Page.objects.get(supertag=supertag, org=get_org())
        except Page.DoesNotExist:
            pass

    @property
    def page(self):
        """
        Пытается опознать страницу, для которой был вызван экшн, используя атрибут page_path или ключ __url__ в query

        @rtype: Page
        @raises: Http404
        """

        if getattr(self, '_page', None):
            return self._page

        # page_path проставляет форматтер, когда вызывает экшен с вики-контекстом.
        # поэтому все вызовы экшенов через форматтер получают тег страницы.
        self._page = self._get_page(self.page_path)
        if self._page:
            return self._page

        # Try to get page by __url__ param. This way works for Http based actions.
        if self.params.get('__url__'):
            tag = urlparse(self.params['__url__']).path.strip('/')

            self._page = self._get_page(tag)
            if self._page:
                return self._page

            error_message = 'No page for action {0} ({1})'
        else:
            error_message = 'Action was called without __url__ parameter {0} ({1})'

        raise Http404(error_message.format(self.name, self.request.get_full_path()))


class WikiActionWithPOST(OldDjangoWikiAction):
    """
    Принимает POST запрос в старой верстке. В nodejs-верстке POST запрос не принимает.

    DEPRECATED. После закрытия старой верстки надо удалить.
    """

    http_method_names = ['get', 'post']
    csrf_exempt = False

    # django.http.HttpRequest instance.
    request = None

    # Does action require authorization?
    login_required = True

    # Specifies page access type to use the action.
    access_required = 'read'

    # Override BaseAction's value by False because complicated actions
    # based on DynamicAction most often display errors in own templates.
    display_errors = False

    # Type of action's messages (not errors).
    # Could be 'local' or 'global'.
    message_type = 'local'

    def __init__(self, params=None, request=None, **kwargs):
        super(WikiActionWithPOST, self).__init__(params, request, **kwargs)
        self.message_type = self.params.get('__message_type__', self.message_type)

        self.request = request
        self.is_json_rendering = False

        # FIXME: костыль! убрать, как избавимся от старого фронтэнда
        if hasattr(request, 'page') and not self.page_path:
            self._page = request.page

    def handle_action(self):
        OldDjangoWikiAction._parse_http_params(self)
        return self.handle()

    def render(self):
        if hasattr(self, 'render_json'):
            if self.request.path.startswith('/_api/frontend/'):
                # При запросе через API нужно отдать приоритет рендерингу блоков в JSON, а не HTML
                # При этом API само делает проверку на аутентификацию и доступ к странице
                self.is_json_rendering = True

        if self.login_required:
            if not self.user:
                self.params['errors'].append(gettext('actions.base_YouHaveToLoginToUseAction'))

        if self.access_required:
            # у request может не быть from_yandex_server, если request надули
            # фальшивый на генераторе, чтобы отформатировать страницу и форматтер
            # вызывает экшен на этой странице.
            if not (
                getattr(self.request, 'from_yandex_server', False)
                or self.page.has_access(user=self.user, privilege=self.access_required)
            ):
                self.params['errors'].append(gettext('actions.base:NoAccessToPage'))
        if self.params['errors']:
            return self._render_errors()

        if self.is_json_rendering:
            # если надо возвращать json, то контекст для теплэйтов нам не нужен
            self.http_content_type = 'application/json'
            return self.render_json()
        else:
            self.http_content_type = 'text/html'

        ctx = self._get_ctx()
        if ctx['errors'] and self.display_errors:
            return self._render_errors()

        self._update_ctx(ctx)

        if not self.template_name:
            tpl_name = 'actions/%s.html' % self.name
        else:
            tpl_name = self.template_name

        rendered_string = loader.render_to_string(tpl_name, ctx, request=self.request)

        # оборачиваем отрендеренное в тэги, обозначающие динамический экшн
        return ''.join((START_DYNAMIC_TAG, rendered_string, END_DYNAMIC_TAG))

    def add_global_message(self, message, status=messages.SUCCESS):
        messages.add_message(self.request, status, message)

    def add_message(self, message):
        if self.message_type != 'local':
            return self.add_global_message(message)
        k = self.name + str(self.number)
        if '_action_messages_' not in self.request.session:
            self.request.session['_action_messages_'] = {}
        if k not in self.request.session['_action_messages_']:
            self.request.session['_action_messages_'][k] = []
        self.request.session['_action_messages_'][k].append(message)
        self.request.session.modified = True

    def get_messages(self):
        if hasattr(self.request, 'session') and '_action_messages_' in self.request.session:
            k = self.name + str(self.number)
            if k in self.request.session['_action_messages_']:
                self.request.session.modified = True
                return self.request.session['_action_messages_'].pop(k, [])
        return []

    def redirect(self, url=None, cookies=None):
        if url is None:
            url = self.page.url + '#' + self.name + str(self.number)
        response = HttpResponseRedirect(url)
        if cookies:
            for cookie, value in cookies.items():
                response.set_cookie(cookie, value)
        return response

    @property
    def user(self):
        if hasattr(self.request, 'user'):
            return self.request.user
        return None

    def _update_ctx(self, ctx):
        super(WikiActionWithPOST, self)._update_ctx(ctx)
        try:
            ctx['action_page'] = self.page
        except (LookupError, Http404, Page.DoesNotExist, APIError):
            pass
        ctx['action_user'] = self.user
        ctx['messages'] = self.get_messages()

    def normalized_supertag(self, tag):
        return translit(canonize(resolve('/' + self.page.supertag, urlparse(tag).path)))

    def normalized_tag(self, tag):
        return canonize(resolve('/' + self.page.tag, urlparse(tag).path)).strip('/')

    def post(self, *args, **kwargs):
        """
        Handle POST request to action.
        """
        self._parse_http_params()
        r = self.handle()
        if isinstance(r, HttpResponse):
            return r

        return self.get(*args, **kwargs)


class OldDjangoWidgetAction(WikiActionWithPOST):
    """
    Виджет экшен для старой верстки.

    Умеет рендерить на своем месте заглушку
    и отвечать версткой для этой заглушки после загрузки страницы в браузер.
    """

    def get(self, *args, **kwargs):
        self._parse_http_params()
        if hasattr(self, 'validate_params'):
            if not self.validate_params():
                return HttpResponse(self._render_errors() if self.display_errors else '', self.http_content_type)
        r = super(OldDjangoWidgetAction, self).render()
        return HttpResponse(r, self.http_content_type, status=self.http_status_code)

    def render(self, *args, **kwargs):
        if hasattr(self, 'validate_params'):
            if not self.validate_params():
                if self.display_errors:
                    return self._render_errors(action_body=True)
                else:
                    return ' '

        html = loader.render_to_string(
            'actions/_widget_action.html', {'action': self.name, 'params': self._urlencoded_params}
        )

        # оборачиваем отрендеренное в тэги, обозначающие динамический экшн
        return ''.join((START_DYNAMIC_TAG, html, END_DYNAMIC_TAG))

    @property
    def _urlencoded_params(self):
        params = self.params.copy()
        params['__count__'] = self.number
        if 'errors' in params:
            del params['errors']
        for i in range(len(self.ordered_params)):
            params[str(i)] = self.ordered_params[i]
        return urlencode(params)


class SimpleWikiAction(OldDjangoWikiAction):
    template_name = 'actions/_http_action.html'
