
import logging
from urllib.parse import urlencode

from django.urls import reverse

from wiki.actions.classes.base_action import ActionError, WikiBaseAction
from wiki.utils.errors import InputValidationError

log = logging.getLogger(__name__)


class WikiWidgetAction(WikiBaseAction):
    """
    Виджет экшен для nodejs-верстки.

    Умеет отдавать плейсхолдер на операцию view форматтера.
    Фронтэнд умеет работать с этим плейсхолдером и заполняет его контентом.

    Возвращает заглушку на операцию view форматтера в json_for_formatter_view.

    {
      "action_name": имя экшена,
      "url_with_data": урл, который нужно вызвать, чтобы получить ответ с данными.
      "params": {"param1": "value1"}
    }

    Параметры с которыми вызван экшен передаются в bemjson.
    Фронтэнд дергает GET url_with_data, за которым отвечает метод get,
    отвечающий реальными данными.

    Из экшена можно бросать ошибки-наследники RestApiError.
    """

    @property
    def action_name(self):
        return str(self.__class__.__name__.lower())

    @staticmethod
    def build_url_with_data(action_name, page_tag, params):
        """
        URL, из которого браузер может забрать контент для отрисовки экшена.
        """
        url = reverse('frontend:pages:call_action', kwargs={'tag': page_tag})
        # тут нужен хак, потому что текущая схема урлов из-за tag-regex
        # создает вот такой урл
        # /_api/frontend/tag.action_view. А нам нужен .../tag/.action_view
        url = '/.'.join(url.rsplit('.', 1))

        url += '?action_name=' + action_name

        if params:
            # Не используйте urlencode с doseq=True в этом месте, т.к. будут неправильно закодированы unicode строки
            url += '&' + urlencode(params)
        return url

    def json_for_formatter_view(self, page, user, params):
        """
        Вернуть JSON для виджет экшенов.

        Выглядит так:
        {
            "action_name": "grid",
            "url_with_data": "/_api/...../?params",
            "params": {params_dict}
        }

        @rtype: dict
        """
        frontend_params = {'action_name': self.action_name}
        frontend_params['params'] = params

        frontend_params.update({'url_with_data': self.build_url_with_data(self.action_name, page.tag, params)})
        return frontend_params

    def render_json_for_get(self, params):
        try:
            return self.json_for_get(params)
        except InputValidationError as exc:
            raise ActionError(errors_or_message=str(exc))

    def json_for_get(self, params):
        """
        Для отображения ошибок нужно вызывать self.error_happened(err_message).

        @type params: dict
        @param params: не ParamsWrapper, а именно dict.
        """
        raise NotImplementedError()
