from django.conf import settings
from django.forms.widgets import Textarea
from django.utils.translation import ugettext

from wiki.actions.classes.form_elements.fields import (
    ClearBooleanField,
    ClearCharField,
    ClearChoiceField,
    ClearDateField,
    ClearFileField,
    ClearFloatField,
    ClearGroupField,
    ClearInfoField,
    ClearMultipleChoiceField,
    ClearServiceField,
    ClearStaffField,
    ClearTicketField,
)
from wiki.actions.classes.form_elements.widgets import (
    ClearGroupAutocomplete,
    ClearInformation,
    ClearServiceAutocomplete,
    ClearStaffAutocomplete,
    ClearStaffAutocompleteMultiple,
)
from wiki.grids.utils.base import ticket_field_names

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)

CACHE_KEYS = {
    'type': 'forms_builder_issue_types',
    'priority': 'forms_builder_issue_priority',
    'status': 'forms_builder_issue_status',
}

DEFAULT_CACHE_TIMEOUT = 60 * 60  # 1 hour


def __parse_field(init_opts, **kwargs):
    """
    Вытащить из описания поля общие параметры которые передаем конструктору
    """
    if kwargs['type'] == 'info':
        # WIKI-6632 — info не может быть required
        init_opts['required'] = False
    else:
        init_opts['required'] = kwargs.get('required', False)

    if 'title' in kwargs:
        init_opts['label'] = kwargs['title']

    if 'text' in kwargs:
        if kwargs['type'] == 'info':
            init_opts['initial'] = kwargs['text']
        else:
            init_opts['help_text'] = kwargs['text']

    if 'default' in kwargs:
        init_opts['initial'] = kwargs['default']

    if 'check' in kwargs and 'string_type' in kwargs['check']:
        init_opts['string_type'] = kwargs['check']['string_type']

    return init_opts


def __parse_field_number(name, **kwargs):
    """Распарсить поле input type="text" с условием ввода числа"""
    call_opts = {}
    if 'check' in kwargs:
        call_opts['min_value'] = kwargs['check']['min'] if 'min' in kwargs['check'] else None
        call_opts['max_value'] = kwargs['check']['max'] if 'max' in kwargs['check'] else None
    init_opts = __parse_field(call_opts, **kwargs)
    return ClearFloatField(**init_opts)


def __parse_field_input_text(name, **kwargs):
    """
    Распарсить поле input type="text"
    """
    call_opts = {}
    if 'check' in kwargs and isinstance(kwargs['check'], dict):
        call_opts['min_length'] = kwargs['check']['min'] if 'min' in kwargs['check'] else None
        call_opts['max_length'] = kwargs['check']['max'] if 'max' in kwargs['check'] else None
    init_opts = __parse_field(call_opts, **kwargs)
    return ClearCharField(**init_opts)


def __parse_field_base_select(name, options=None, subs=None, first=None, multiple=False, nochoicecheck=False, **kwargs):
    init_opts = __parse_field({}, **kwargs)
    init_opts['choices'] = options if options else []
    init_opts['substituted_values'] = subs if subs else {}
    if first:
        # dont create copy of link to prepend
        init_opts['choices'].reverse()
        init_opts['choices'].append(('', first))
        init_opts['choices'].reverse()
        init_opts['substituted_values'][''] = ugettext('not selected')
        init_opts['initial'] = ''
    return init_opts


def __parse_field_select(*args, **kwargs):
    """ Распарсить поле типа список """
    init_opts = __parse_field_base_select(*args, **kwargs)
    if kwargs.get('multiple'):
        return ClearMultipleChoiceField(**init_opts)
    return ClearChoiceField(**init_opts)


def __parse_field_staff(name, **kwargs):
    """
    Создать выпадающий список с персоналом
    """
    multiple = kwargs.get('multiple', False)
    if multiple:
        widget = ClearStaffAutocompleteMultiple(multiple=multiple)
    else:
        widget = ClearStaffAutocomplete(multiple=multiple)

    init_opts = __parse_field(
        {
            'widget': widget,
            'show': kwargs.get('show', None),
            'multiple': multiple,
        },
        **kwargs
    )
    return ClearStaffField(**init_opts)


def __parse_field_group(name, **kwargs):
    init_opts = __parse_field(
        {'widget': ClearGroupAutocomplete, 'group_type': kwargs.get('group', 'departments')}, **kwargs
    )
    return ClearGroupField(**init_opts)


def __parse_field_service(name, **kwargs):
    init_opts = __parse_field(
        {
            'widget': ClearServiceAutocomplete,
            'show': kwargs.get('show', None),
        },
        **kwargs
    )
    return ClearServiceField(**init_opts)


def __parse_field_info(name, **kwargs):
    """
    Создать поясняющий текст
    """
    init_opts = __parse_field({'widget': ClearInformation}, **kwargs)
    init_opts.update(value=kwargs.pop('value', None))
    return ClearInfoField(**init_opts)


__parse_field_passthrough = __parse_field_info


def __parse_field_textarea(name, **kwargs):
    """
    Создать поле для ввода текста
    """
    init_opts = __parse_field({}, **kwargs)
    init_opts['widget'] = Textarea(attrs={'rows': 3})
    return ClearCharField(**init_opts)


def __parse_field_checkbox(name, **kwargs):
    """
    Создать чекбокс
    """
    init_opts = __parse_field({}, **kwargs)
    return ClearBooleanField(**init_opts)


def __parse_field_file(name, **kwargs):
    """
    Прицепить поле типа файл
    """
    init_opts = __parse_field({}, **kwargs)
    if 'check' in kwargs:
        if 'accept' in kwargs['check']:
            init_opts['accept'] = (
                [kwargs['check']['accept']] if isinstance(kwargs['check']['accept'], str) else kwargs['check']['accept']
            )
        else:
            init_opts['accept'] = None
    return ClearFileField(**init_opts)


def __parse_ticket_field(*args, **kwargs):
    init_opts = __parse_field({}, **kwargs)
    return ClearTicketField(**init_opts)


def __parse_field_date(name, **kwargs):
    """
    Создать поле "дата"
    """
    init_opts = __parse_field({}, **kwargs)
    return ClearDateField(**init_opts)


FIELD_HANDLERS = {
    'string': __parse_field_input_text,
    'text': __parse_field_textarea,
    'select': __parse_field_select,
    'staff': __parse_field_staff,
    'info': __parse_field_info,
    'date': __parse_field_date,
    'checkbox': __parse_field_checkbox,
    'file': __parse_field_file,
    'number': __parse_field_number,
}

if IS_INTRANET:
    FIELD_HANDLERS['group'] = __parse_field_group
    FIELD_HANDLERS['service'] = __parse_field_service

if settings.ISSUE_TRACKERS_ENABLED:

    FIELD_HANDLERS['ticket'] = __parse_ticket_field

    # Don't validate other ticket fields in any way.
    for key in (
        ticket_field_names.subject,
        ticket_field_names.assignee,
        ticket_field_names.reporter,
        ticket_field_names.status,
        ticket_field_names.type,
        ticket_field_names.priority,
        ticket_field_names.updated_at,
        ticket_field_names.created_at,
        ticket_field_names.original_estimation,
        ticket_field_names.fix_versions,
        ticket_field_names.components,
        ticket_field_names.story_points,
    ):
        FIELD_HANDLERS[key] = __parse_field_passthrough
