"""
Поля формы, которые позволяют jQuery использовать валидацию.

@author: chapson
@status: Development
@maintainer: Chaporgin Anton
@copyright: 2010, Yandex
@version: 0.0
"""
import re
from urllib.parse import quote

from django.core import validators
from django.core.exceptions import ValidationError
from django.forms.fields import (
    BooleanField,
    CharField,
    ChoiceField,
    ComboField,
    DateField,
    DateTimeField,
    DecimalField,
    EmailField,
    FileField,
    FilePathField,
    FloatField,
    GenericIPAddressField,
    ImageField,
    IntegerField,
    MultipleChoiceField,
    MultiValueField,
    NullBooleanField,
    RegexField,
    SlugField,
    SplitDateTimeField,
    TimeField,
    TypedChoiceField,
    URLField,
)
from django.forms.widgets import PasswordInput, Textarea, TextInput
from django.utils.translation import get_language
from django.utils.translation import ugettext_lazy as _
from wiki.intranet.models import Staff
from ids.exceptions import IDSException

from wiki.org import org_group, org_staff
from wiki.users.models import Group
from wiki.utils import planner


def if_is_required(field, result):
    """
    Если поле обязательно, то добавить к атрибуту класс строку required
    """
    if field.required:
        result['class'] = result['class'] + ' required' if 'class' in result else 'required'
    return result


def common_widget_attrs(result, field, widget):
    """
    Установить общие правила валидации для всех виджетов
    """
    if hasattr(field, 'string_type') and field.string_type is not None:
        result.update({field.string_type: 'true'})


def common_kwargs_params(field, kwargs):
    """
    Установить тип вводимого значения для текстовых полей

    Фактически это подсказка для валидатора на стороне клиента
    """
    if 'string_type' not in kwargs:
        return

    string_type = kwargs.pop('string_type')
    setattr(field, 'string_type', string_type)
    validator = {
        'email': validators.validate_email,
        'url': validators.URLValidator(verify_exists=False, validator_user_agent=validators.URL_VALIDATOR_USER_AGENT),
    }.get(string_type)

    check_validator = lambda v: isinstance(validator, v.__class__)
    if validator and not any(filter(check_validator, field.default_validators)):
        field.default_validators.append(validator)


class ClearCharField(CharField):
    def __init__(self, choices=None, substituted_values=None, **kwargs):
        if choices:
            self.choices = choices
        self.default_validators = []
        common_kwargs_params(self, kwargs)
        super(ClearCharField, self).__init__(**kwargs)

    choices = None

    def widget_attrs(self, widget):
        result = super(ClearCharField, self).widget_attrs(widget) or {}
        common_widget_attrs(result, self, widget)
        result = if_is_required(self, result)
        if self.min_length is not None and isinstance(widget, (TextInput, PasswordInput)):
            result['minlength'] = self.min_length
        if 'class' in result:
            result['class'] += ' b-wiki-form__wide'
        else:
            result['class'] = ' b-wiki-form__wide'
        if isinstance(widget, Textarea):
            result['class'] = result['class'] + ' b-autoarea g-js' if 'class' in result else 'b-autoarea g-js'
            result['onclick'] = 'return {name: "b-autoarea"}'
        return result


tracker_ticket_re = re.compile(r'^\w+-\d+$')


def validate_tracker_ticket(value):
    try:
        if tracker_ticket_re.match(value) is not None:
            return
    except TypeError:
        pass
    raise ValidationError(_('actions.forms:Tracker ticket is invalid %s') % value)


class ClearTicketField(CharField):
    def __init__(self, **kwargs):
        self.default_validators = [validate_tracker_ticket]
        common_kwargs_params(self, kwargs)
        super(ClearTicketField, self).__init__(**kwargs)


class ClearGroupField(ClearCharField):
    def __init__(self, group_type='departments', **kwargs):
        if isinstance(group_type, str):
            self.group_type = [group_type]
        else:
            self.group_type = group_type
        common_kwargs_params(self, kwargs)
        super(ClearGroupField, self).__init__(**kwargs)

    def widget_attrs(self, widget):
        result = super(ClearGroupField, self).widget_attrs(widget) or {}
        common_widget_attrs(result, self, widget)
        result = if_is_required(self, result)
        result['onclick'] = 'return {type: [%s]}' % ','.join(("'%s'" % quote(_) for _ in self.group_type))
        return result

    def transform_value(self, value):
        try:
            value = org_group().get(url=value)
        except Group.DoesNotExist:
            pass
        if isinstance(value, Group):
            chain = list(value.get_ancestors()[1:]) + [value]
            return ' / '.join(group.name for group in chain)
        return value


def staff_by_login(func):
    def _staff(value, *args, **kwargs):
        __staff = 1
        if __staff:
            return func(__staff, *args, **kwargs)
        else:
            raise Exception('login ld "%s" not found' % value)

    return _staff


PROPOSED_STAFF_ATTRS = [
    'login',
    'domain',
    'first_name',
    'middle_name',
    'last_name',
    'first_name_en',
    'last_name_en',
    'en_name',
    'birthday',
    'family_status',
    'children',
    'car',
    'car_num',
    'address',
    'edu_place',
    'edu_date',
    'office',
    'join_at',
    'desk_id',
    'quit_at',
    'employment',
    'work_phone',
    'work_email',
    'mobile_phone',
    'department',
    'gender',
    'is_dismissed',
]

real_staff_attrs = [f.name for f in Staff._meta.fields]

ALLOWED_STAFF_ATTRS = [f for f in PROPOSED_STAFF_ATTRS if f in real_staff_attrs] + [
    'all_groups',
    'email',
]


class ClearStaffField(ClearCharField):
    def __init__(self, show=None, **kwargs):
        self.show = show and [el for el in show.split() if el in ALLOWED_STAFF_ATTRS]
        self.multiple = kwargs.pop('multiple', False)
        common_kwargs_params(self, kwargs)
        super(ClearStaffField, self).__init__(**kwargs)

    def widget_attrs(self, widget):
        result = super(ClearStaffField, self).widget_attrs(widget) or {}
        common_widget_attrs(result, self, widget)
        result = if_is_required(self, result)
        return result

    def get_staff(self, value):
        """
        Получить всех людей из стаффа, которых можно получить

        Остальные логины добавляются в конец спиcка
        @param value: строка логинов (можно перечислить через запятую)
        """
        logins = [login_str.strip() for login_str in value.split(',')]
        persons = list(self._resolve_logins(logins))
        unresolved_logins = set(logins) - set(p.login for p in persons)
        return persons + list(unresolved_logins)

    def _resolve_logins(self, logins):
        """
        Получить список людей из сервиса Стафф на основании их логинов

        @param logins: список логинов
        """
        return org_staff().filter(login__in=logins)

    def transform_value(self, value):
        """
        Преобразовать значения к нужному виду

        @return: string
        """
        result = []
        for staff in self.get_staff(value):
            if isinstance(staff, Staff):
                answer = []
                attrs = self.show or ['first_name', 'last_name']
                for attr in attrs:
                    if attr == 'email':
                        answer.append(staff.get_email())
                    elif hasattr(staff, attr):
                        item = getattr(staff, attr)
                        if not isinstance(item, str) and getattr(item, '__iter__', False):
                            answer.append(' / '.join([str(el) for el in item]))
                        else:
                            answer.append(str(item))
                result.append(' '.join(answer))
        return ','.join(result) if result else value


class ClearIntegerField(IntegerField):
    def __init__(self, max_value=None, min_value=None, **kwargs):
        self.max_value, self.min_value = max_value, min_value
        super(ClearIntegerField, self).__init__(max_value=max_value, min_value=min_value, **kwargs)

    def widget_attrs(self, widget):
        result = super(ClearIntegerField, self).widget_attrs(widget) or {}
        result = if_is_required(self, result)
        result['number'] = 'true'
        return result


class ClearFloatField(FloatField):
    def __init__(self, max_value=None, min_value=None, **kwargs):
        self.max_value, self.min_value = max_value, min_value
        super(ClearFloatField, self).__init__(max_value=max_value, min_value=min_value, **kwargs)

    def to_python(self, value):
        if isinstance(value, str):
            value = value.replace(',', '.')
        return super(ClearFloatField, self).to_python(value)

    def widget_attrs(self, widget):
        result = super(ClearFloatField, self).widget_attrs(widget) or {}
        result = if_is_required(self, result)
        if self.max_value is not None:
            result['max'] = self.max_value
        if self.min_value is not None:
            result['min'] = self.min_value
        result['number'] = 'true'
        return result


class ClearDecimalField(DecimalField):
    def __init__(self, max_value=None, min_value=None, *args, **kwargs):
        self.max_value, self.min_value = max_value, min_value
        super(ClearDecimalField, self).__init__(max_value=max_value, min_value=min_value, **kwargs)

    def widget_attrs(self, widget):
        result = super(ClearDecimalField, self).widget_attrs(widget) or {}
        if_is_required(self, result)
        result['number'] = 'true'
        return result


class ClearDateField(DateField):
    def widget_attrs(self, widget):
        result = super(ClearDateField, self).widget_attrs(widget) or {}
        result = if_is_required(self, result)
        result['class'] = result['class'] + ' is_date' if 'class' in result else 'is_date'
        return result


class ClearTimeField(TimeField):
    pass


class ClearDateTimeField(DateTimeField):
    pass


class ClearRegexField(RegexField):
    pass


class ClearEmailField(EmailField):
    pass


class ClearFileField(FileField):
    def __init__(self, accept=None, **kwargs):
        self.accept = accept
        super(ClearFileField, self).__init__(**kwargs)

    def widget_attrs(self, widget):
        result = super(ClearFileField, self).widget_attrs(widget) or {}
        result = if_is_required(self, result)
        if self.accept is not None:
            result['accept'] = ' '.join(self.accept)
        return result


class ClearImageField(ImageField):
    pass


class ClearURLField(URLField):
    pass


class ClearBooleanField(BooleanField):
    def widget_attrs(self, widget):
        self.is_checkbox = True
        result = super(BooleanField, self).widget_attrs(widget) or {}
        result = if_is_required(self, result)
        return result


class ClearNullBooleanField(NullBooleanField):
    pass


def __choice_transform_value(self, value):
    value = value if not isinstance(value, str) and hasattr(value, '__iter__') else [value]
    _v = []
    for v in value:
        if v in self.substituted_values:
            _v.append(str(self.substituted_values[v]))
        else:
            for test_value, substitute in self.choices:
                if test_value == v:
                    _v.append(str(substitute))
                    break
    return ','.join(_v)


class ClearChoiceField(ChoiceField):
    def __init__(self, substituted_values=None, **kwargs):
        self.substituted_values = substituted_values or []
        ChoiceField.__init__(self, **kwargs)

    def widget_attrs(self, widget):
        result = super(ClearChoiceField, self).widget_attrs(widget) or {}
        common_widget_attrs(result, self, widget)
        result = if_is_required(self, result)
        return result


ClearChoiceField.transform_value = __choice_transform_value


class ClearTypedChoiceField(TypedChoiceField):
    pass


class ClearMultipleChoiceField(MultipleChoiceField):
    def __init__(self, substituted_values=None, **kwargs):
        self.substituted_values = substituted_values or []
        ChoiceField.__init__(self, **kwargs)

    def widget_attrs(self, widget):
        result = super(MultipleChoiceField, self).widget_attrs(widget) or {}
        common_widget_attrs(result, self, widget)
        result = if_is_required(self, result)
        return result


ClearMultipleChoiceField.transform_value = __choice_transform_value


class ClearComboField(ComboField):
    pass


class ClearMultiValueField(MultiValueField):
    pass


class ClearFilePathField(FilePathField):
    pass


class ClearSplitDateTimeField(SplitDateTimeField):
    pass


class ClearIPAddressField(GenericIPAddressField):
    pass


class ClearSlugField(SlugField):
    pass


class ClearInfoField(CharField):
    """
    Поле показывающее пояснительную информацию к форме
    """

    def __init__(self, **kwargs):
        self.value = kwargs.pop('value', None)
        super(ClearInfoField, self).__init__(**kwargs)

    def clean(self, value):
        return self.value


class ClearServiceField(ClearCharField):
    AVAILABLE_SERVICE_SHOW_FIELDS = (
        'id',
        'name',
        'owner',
    )

    default_error_messages = {
        'invalid': _('Invalid service id'),
    }

    def __init__(self, show=None, **kwargs):
        if show in self.AVAILABLE_SERVICE_SHOW_FIELDS:
            self.show = show
        else:
            self.show = 'name'
        self.services = planner.get_planner_services_repository()
        super(ClearServiceField, self).__init__(**kwargs)

    def widget_attrs(self, widget):
        result = super(ClearServiceField, self).widget_attrs(widget) or {}
        common_widget_attrs(result, self, widget)
        result = if_is_required(self, result)
        result['onclick'] = "return { hidden: 'id' }"
        return result

    def to_python(self, value):
        if not value:
            return

        try:
            id = int(value)
        except ValueError:
            raise ValidationError(self.error_messages['invalid'])

        try:
            service = self.services.get_one(
                {
                    'id': id,
                    'fields': ','.join(self.AVAILABLE_SERVICE_SHOW_FIELDS),
                }
            )
        except IDSException as exc:
            if exc.status_code == 404:
                raise ValidationError(self.error_messages['invalid'])
            raise ValidationError('. '.join([str(self.error_messages['invalid']), str(exc)]))
        return service

    def transform_value(self, value):
        if value is None:
            return ''

        service = value
        if self.show == 'id':
            return service['id']
        if self.show == 'name':
            user_language = get_language()
            return service['name'].get(user_language, 'ru')
        if self.show == 'owner':
            return 'staff:' + service['owner']['login']
        return ''
