from django.utils.translation import ugettext_lazy as _
from wiki.legacy.choices import Choices


class FormHandlingError(Exception):
    """
    Ошибка при обработке отправленной формы (не валидация отправленных данных).
    """

    CODES = Choices(
        GENERIC=1,  # форма валидна, ошибка при обработке, например отправке письма
        MISSING_PAGE=2,
    )

    def __init__(self, code, arg=None, message=None):
        """
        :param code: код ошибки, ключ из CODES (для нового фронтенда)
        :param arg: строка с идентификацией проблемной сущности, например тэг несуществующей страницы
        :param message: строка с сообщением об ошибке
        """
        super(FormHandlingError, self).__init__(message)
        self.code = code
        self.arg = arg


class FormSubmitValidationError(Exception):
    """
    Ошибка валидации отправленной формы.
    """

    def __init__(self, message, errors=None, form=None):
        """
        :param message: строка с сообщением об ошибке (для старого фронтенда и для логов)
        :param errors: словарь строка-строка с деталями ошибки
        :param form: инстанс формы (нужно только для старого фронтенда)
        """
        super(FormSubmitValidationError, self).__init__(message)
        self.errors = errors
        self.form = form


# изменяя VALIDATION_SOURCE_CODES изменяй и VALIDATION_TRANSLATIONS.
VALIDATION_SOURCE_CODES = Choices(
    YAML=1,
    TARGET_FIELD_NOT_FOUND=2,
    INVALID_TARGET_ENTRY=3,
    STARTREK_FIELDS_ERROR=4,
)

VALIDATION_TRANSLATIONS = {
    # Translators:
    #   ru: Для формы указан неверный YAML
    #   en: Invalid YAML given for the form
    VALIDATION_SOURCE_CODES.YAML: _('Invalid YAML given'),
    # Translators:
    #   ru: Не найдено поле формы, указанное в target
    #   en: Could not find form field, specified in target
    VALIDATION_SOURCE_CODES.TARGET_FIELD_NOT_FOUND: _('Target cannot be found'),
    # Translators:
    #   ru: Поле target в YAML содержит некорректное значение
    #   en: Target field in YAML has an incorrect value
    VALIDATION_SOURCE_CODES.INVALID_TARGET_ENTRY: _('Target field is not correct'),
    # Translators:
    #   ru: В полях startrek есть ошибки
    #   en: There are errors in startrek fields
    VALIDATION_SOURCE_CODES.STARTREK_FIELDS_ERROR: _('There are errors in startrek fields'),
}


class FormSourceValidationError(Exception):
    """
    Ошибка разбора исходного текста формы.
    """

    CODES = VALIDATION_SOURCE_CODES

    def __init__(self, code, message):
        """
        :param code: код ошибки, ключ из CODES (для нового фронтенда)
        :param message: строка с сообщением об ошибке (для старого фронтенда и для логов)
        """
        super(FormSourceValidationError, self).__init__(message)
        self.code = code

    def to_frontend_message(self):
        """
        Сформировать ответ для фронтэнда

        @rtype: unicode
        """
        if self.code == self.CODES.YAML:
            # библиотека для парсинга YAML форматирует ошибки так, чтобы
            # можно было как-то разобраться где именно произошла ошибка.
            # поэтому мы отображаем ошибку на отдельной строке.
            return '{0}.\n{1}'.format(VALIDATION_TRANSLATIONS[self.code], str(self))
        return '{0}: {1}'.format(VALIDATION_TRANSLATIONS[self.code], str(self))
