"""
Base class for all the nice looking forms
"""

import collections.abc
import functools
import itertools
import logging
import os
from urllib.parse import ParseResult, urlparse

from django import forms
from django.conf import settings
from django.forms.fields import DateField, FileField
from django.utils.translation import ugettext as _
from wiki.actions.classes.form_elements import submit_handlers
from wiki.actions.classes.form_elements.fields import ClearBooleanField, ClearInfoField, ClearStaffField
from wiki.actions.classes.form_elements.form_errors import FormSourceValidationError
from wiki.actions.classes.form_elements.submit_handlers import startrek
from wiki.actions.classes.form_elements.submit_handlers.pagefile import PageFileSubmitter
from wiki.actions.classes.form_elements.target import Target, group_targets, merge_targets
from wiki.legacy.staff_services import get_superior
from wiki.org import org_staff
from wiki.utils import smart_list
from wiki.utils.supertag import translit

logger = logging.getLogger(__name__)
mail_logger = logging.getLogger('mail_logger')


def get_boss(staff):
    if not staff.is_dismissed:
        return get_superior(staff)


def get_department_boss(staff):
    """
    Вернуть руководителя направления (в прошлом - департамента).
    """
    if not getattr(staff, 'is_dismissed', False):
        directions = staff.department.get_ancestors(include_self=True).filter(kind_id=settings.DIS_DIRECTION_KIND_ID)
        if directions:
            return directions[0].chief


handlers_pseudonims = {
    'page': submit_handlers.PageAnchorSubmitHandler,
    'email': submit_handlers.EmailSubmitHandler,
    'http': submit_handlers.HttpSubmitHandler,
    'excel': submit_handlers.ExcelSubmitter,
    'excelpage': submit_handlers.PageFileSubmitter,
}


class ClearForm(forms.Form):
    """
    Базовый класс для реализации формы
    """

    class Media:
        js = (
            '/_js/jquery/jquery.metadata.js',
            '/_js/jquery/jquery.validate.js',
            '/_js/jquery/jquery.translations.js',
            '/_js/jquery/init.js',
        )
        css = {'all': ('/_css/forms.css',)}

    _targets = None
    _all_startrek_fields = None

    def __init__(
        self,
        target=None,
        title='',
        again=False,
        submit=None,
        html=False,
        success='',
        date_format=None,
        template=None,
        subject=None,
        startrek_fields=None,
        url=None,
        **kwargs
    ):
        self.form_title = title
        self.target = target or url or []
        self.handler_options = {
            'template': template,
            'html': html,
            'subject': subject,
        }
        self.again = again
        self.success_message = success
        self.submit_button = submit or _('Send')
        self.url = url
        # чтобы изменить формат вывода дат, будет использован как в клиентской
        # части так и рендерами всякими
        self.date_format = date_format or '%Y-%m-%d'
        self.startrek_fields = startrek_fields
        super(ClearForm, self).__init__(**kwargs)

    @property
    def has_upload(self):
        """
        Узнать что форма может передавать файлы
        """
        for f in self.fields.values():
            if isinstance(f, FileField):
                return True
        return False

    def save(self):
        """
        Разослать всем кому надо, посохранять, в общем, сделать дело!
        """
        # фиксируем адресатов для разбора разных ситуаций (WIKI-4048)
        msg = "Form submission by %s; \"%s\"; \"%s\". Targets: %s"
        mail_logger.info(
            msg
            % (
                self.user.username,
                self.url,
                repr(self.target),
                ', '.join(map(str, self.targets)),
            )
        )

        try:
            self.handle_targets()
        finally:
            self.forget_files()

    @property
    def targets(self):
        if self._targets is None:
            self._targets = self.parse_targets()
        return self._targets

    def check_structure(self):
        """
        raises FormSourceValidationError if form structure is invalid
        """
        self.parse_targets()

    def parse_targets(self):
        """
        Дорогих операций тут нет, можно использовать для проверки таргетов.
        """
        _targets = []

        if isinstance(self.target, str):
            # нам передана некоторая строка, например "thasonic@yandex-team.ru, /Homepage"
            # в этом случае мы объединяем цели по типам (например отправляем одно письмо
            # со множеством адресатов, если их перечислено несколько), потому что так
            # было изначально реализовано и пользователи ожидают, что будет так.
            target_entries = [e.strip() for e in self.target.split(',')]
            parsed = list(map(self.parse_target_entry, target_entries))
            parsed_list = list(itertools.chain.from_iterable(smart_list(p) for p in parsed if p))
            for type, targets in group_targets(targets=parsed_list):
                _targets.append(merge_targets(targets))
        else:
            # нам передан список строк и/или маппингов — каждый такой таргет
            # будет обрабатываться отдельно, потому что у них могут быть параметры
            # и слеплять их будет кажется не тем, что ожидает пользователь.
            target_entries = self.target

            for entry in target_entries:
                if isinstance(entry, str):
                    parsed = self.parse_target_entry(entry)
                    parsed_list = smart_list(parsed)
                    _targets.extend(parsed_list)
                elif isinstance(entry, collections.abc.Mapping):
                    if 'to' not in entry:
                        message = _('`to` key must be present')
                        raise FormSourceValidationError(
                            code=FormSourceValidationError.CODES.INVALID_TARGET_ENTRY,
                            message=message,
                        )

                    target_params = dict(entry)
                    to = target_params.pop('to', None)
                    parsed = self.parse_target_entry(entry=to)
                    parsed_list = smart_list(parsed)
                    if parsed:
                        _targets.extend([Target(to=t.to, type=t.type, **target_params) for t in parsed_list])

        return _targets

    def handle_targets(self):
        for target in self.targets:
            if target.type not in handlers_pseudonims:
                continue
            handlers_cls = handlers_pseudonims[target.type]
            handler = handlers_cls(form=self, **self.handler_options)
            handler.handle(target)

    def forget_files(self):
        """
        Удалить временные файлы, которые django размещала на диске при работе
        django.core.files.uploadhandler.TemporaryFileUploadHandler
        (создаются лишь для больших файлов, мелкие - в памяти)
        """

        for f in self.files.values():
            if hasattr(f, 'temporary_file_path') and os.path.exists(f.temporary_file_path()):
                try:
                    os.remove(f.temporary_file_path())
                except Exception as exc:
                    logger.exception(exc)

    def parse_target_entry(self, entry, called_recursively=False):
        """
        Обработать один из элементов поля target.
        @rtype: Target instance or list [<Target>, <Target>]

        Здесь могут быть
        * почтовый адрес (thasonic@yandex-team.ru)
        * ключевое слово staff - уходит письмо заполняющему форму (%staff%)
        * поле из которого взять значения, напр. project_assignee
        * url, чтобы вставить текст в вики после якоря
          WackoWiki/vodstvo/actions/KonstruktorForm#Validator
        * ключ очереди в трекере
        """

        # special target matchers
        def is_staff_target(target):
            return '%staff%' == target

        def is_boss_target(target):
            return '%boss%' == target

        def is_department_boss_target(target):
            return '%department-boss%' == target

        def is_form_field_target(target):
            return target[1:] in self.fields

        # special target parsers
        def entry_boss(target):
            boss = get_boss(self.user.staff)
            if boss:
                return Target(type='email', to=boss.get_email())

        def entry_staff(target):
            return Target(type='email', to=self.user.staff.get_email())

        def entry_department_boss(target):
            boss = get_department_boss(self.user.staff)
            if boss:
                return Target(type='email', to=boss.get_email())

        def entry_form_field(target):
            if called_recursively:
                return

            if not self.is_valid():
                return

            field_name = target[1:]

            values = self.cleaned_data.get(field_name).split(',')
            values = [v.strip() for v in values]
            if isinstance(self.fields[field_name], ClearStaffField):
                persons = org_staff().filter(login__in=values)
                return Target(type='email', to=[person.get_email() for person in persons])
            else:
                parse = functools.partial(self.parse_target_entry, called_recursively=True)
                targets = list(map(parse, values))
                targets = list(itertools.chain.from_iterable(smart_list(p) for p in targets if p))
                merged_targets = []
                for type, targets in group_targets(targets=targets):
                    merged_target = merge_targets(targets)
                    merged_targets.append(
                        Target(
                            to=merged_target.to,
                            type=merged_target.type,
                        )
                    )
                return merged_targets

        # пустой entry допустим, например см. WIKI-6695, просто пропускаем
        if not entry:
            return

        # специальные случаи
        if entry.startswith('%'):
            entry_checks = {
                is_staff_target: entry_staff,
                is_boss_target: entry_boss,
                is_department_boss_target: entry_department_boss,
                is_form_field_target: entry_form_field,
            }
            for matcher, take_action in entry_checks.items():
                if matcher(entry):
                    target = take_action(entry)
                    return target
            else:
                raise FormSourceValidationError(
                    code=FormSourceValidationError.CODES.TARGET_FIELD_NOT_FOUND,
                    message=_('No field found') + (' "%s"' % entry),
                )

        # остальные случаи
        # * EMAIL
        # * excel by email
        # * excel файл у страницы
        # * страница в вики
        # * произвольная url
        # * очередь в трекере
        parsed = urlparse(entry)

        ALLOWED_DOMAINS = ('yandex-team.ru', 'yandex.net', 'yandex.ru')
        if parsed.scheme == 'excel':
            return Target(type='excel', to=parsed.netloc)
        elif parsed.scheme == 'excelpage':
            return Target(type='excelpage', to=entry[len('excelpage://') :])
        elif '@' in parsed.path:
            return Target(type='email', to=entry)
        elif parsed.fragment:
            path = parsed.path or urlparse(self.url).path
            edited_parsed_url = parsed._replace(path=path)
            return Target(type='page', to=edited_parsed_url)
        elif parsed.netloc == '' or parsed.netloc in settings.FRONTEND_HOSTS:
            return Target(type='page', to=parsed)
        elif parsed.hostname.endswith(ALLOWED_DOMAINS):
            return Target(type='http', to=parsed)
        else:
            raise FormSourceValidationError(
                code=FormSourceValidationError.CODES.INVALID_TARGET_ENTRY,
                message=_('No field found') + ' ' + entry,
            )

    def check_fields_compatibility(self, startrek_field, form_field):
        startrek_field_id = startrek_field['id']
        startrek_type_key = startrek.get_startrek_field_type(startrek_field)
        compatible_type_group = startrek.STARTREK_FIELD_TYPES.get(startrek_type_key)

        if compatible_type_group is None:
            message = _('Unsupported Startrek field') + ' ' + startrek_field_id
            raise FormSourceValidationError(
                FormSourceValidationError.CODES.STARTREK_FIELDS_ERROR,
                message=message,
            )

        compatible_form_types = startrek.FORM_FIELD_TYPE_GROUPS[compatible_type_group]

        form_field_type = startrek.get_form_field_type(form_field)
        if form_field_type not in compatible_form_types:
            message = _('Incompatible type for field') + ' ' + startrek_field_id
            raise FormSourceValidationError(
                code=FormSourceValidationError.CODES.STARTREK_FIELDS_ERROR,
                message=message,
            )

    def for_handler(self):
        """
        Затребовать от формы отдать список для вывода [{field,value,name}, ...]
        Используйте вне формы. Результат не отображает ее реальную структуру.

        Пропускаем поля
        * ClearInfoField
        """
        result = []
        for name, field in self.fields.items():
            if name and field:
                if isinstance(field, ClearInfoField):
                    continue
                if isinstance(field.widget, forms.HiddenInput):
                    continue
                value = self.get_cleaned_data(name)
                if isinstance(field, DateField):
                    value = value.strftime(self.date_format) if value else None
                if isinstance(field, ClearBooleanField) and not value:
                    value = ''
                result.append({'field': field, 'value': value, 'name': name})
        return result

    def get_cleaned_data(self, name, default='', transform=True):
        """
        Получить данные с учетом того, что некоторые поля сами определяют
        какое значение выводить

        Обертка над self.cleaned_data.get(...)
        """
        field = self.fields[name]
        value = self.cleaned_data.get(name)
        if transform and hasattr(field, 'transform_value'):
            value = field.transform_value(value)
        return value if value is not None else default

    def fields_required(self):
        return len(self.fields) > 0 and any(field.required for field in self.fields.values())

    def all_fields_required(self):
        return len(self.fields) > 0 and all(field.required for field in self.fields.values())

    def get_page_view_effect(self):
        """
        Определяет, изменяет ли сабмит формы страницу и/или файлы страницы, на которой находится форма.
        Возвращает тапл (bool страница поменялась, bool файлы на странице поменялись).

        Вызывать можно только после того, как отработал parse_targets.
        """
        supertag = translit(urlparse(self.url).path)

        pages_with_affected_texts = self._get_pages_with_affected_texts()
        pages_with_affected_files = self._get_pages_with_affected_files(supertag)
        text_affected = supertag in pages_with_affected_texts
        files_affected = supertag in pages_with_affected_files

        if not files_affected and text_affected:
            for name, field in self.fields.items():
                if isinstance(field, FileField) and self.get_cleaned_data(name) and name in self.files:
                    files_affected = True
                    break

        return text_affected, files_affected

    def _get_pages_with_affected_texts(self):
        def get_page_target_supertag(target):
            if isinstance(target, ParseResult):
                return translit(target.path)
            return translit(target)

        pages_with_affected_text = []
        for target in self.targets:
            if target.type != 'page':
                continue
            to = target.to
            if isinstance(to, ParseResult):
                pages_with_affected_text.append(get_page_target_supertag(to))
            else:
                pages_with_affected_text.extend(list(map(get_page_target_supertag, to)))

        return pages_with_affected_text

    def _get_pages_with_affected_files(self, form_supertag):
        def get_excelpage_target_supertag(target):
            filename, supertag = PageFileSubmitter.split_target(target)
            if supertag is None:
                return form_supertag
            return supertag

        pages = []
        for target in self.targets:
            if target.type != 'excelpage':
                continue
            pages.extend(list(map(get_excelpage_target_supertag, target.to)))

        return pages
