import re
from collections import OrderedDict
from functools import reduce

import yaml
from django.conf import settings

from wiki.actions.classes.form_elements import field_parsers
from wiki.actions.classes.form_elements.fields import ClearGroupField, ClearServiceField, ClearStaffField
from wiki.actions.classes.form_elements.form_errors import FormSourceValidationError
from wiki.actions.classes.form_elements.forms import ClearForm


def yaml_keys_natural_order(declaration, parent_key=None):
    """
    Получить список ключей в порядке перечисления их в конструкции

    Функция не совсем закончена -- не учитывает, что подключи с данным
    отступом могут быть не только внутри данного ключа, но и внутри других
    (подправить второе регулярное выражение).

    Получить список из ключей сложного узла YAML конструкции, которые
    представлены в том порядке, что перечислены в этой конструкции. Если
    найдется больше чем один, то будет использован самый первый ничего не
    поделать. Это поведение легко доделать самостоятельно.

    @param declaration: YAML конструкция
    @param parent_key: родительский ключ, из которого вытащим ключей-детей
    """

    # WIKI-4879 Пробелы на конце строк могут сломать первую регулярку
    # (например, если между template и fields будет
    # строка из двух пробелов), поэтому удалим их.
    # Вообще их можно удалить с самого начала, они ни должны ни на что
    # повлиять, но внутри этой функции это делать совсем безопасно.
    declaration = '\n'.join(lq.rstrip() for lq in declaration.split('\n'))

    result = []
    step = 0
    # learn how many spaces we have to indent to get subkeys
    pattern = r'.*([\s^\n].*:\n)?\n?([\s^\n]*)%s:' % parent_key
    match = re.search(pattern, declaration)
    if match is not None:
        g = match.groups()
        # count the parent of parent indent
        if g[0] is not None:
            step -= len(g[0]) - len(g[0].lstrip())
        indent = len(g[1])
        step += indent
        if step == 0:
            step = 2  # fall to default behavior
        # get all indented values
        keys = re.findall(r'(\n%s(\w*):)' % (' ' * (indent + step),), declaration)
        result = [x[1].strip() for x in keys]
    return result


class OMap(yaml.YAMLObject):
    yaml_tag = '!OMap'
    yaml_loader = yaml.SafeLoader

    @classmethod
    def from_yaml(cls, loader, node):
        return custom_omap_constructor(loader, node)


def custom_omap_constructor(loader, node):
    """
    Пытаюсь сохранить порядок ключей при загрузке кода с помощью PyYaml
    @see: http://pyyaml.org/ticket/29#comment:4

    Копипаст кода из yaml.constructors.BaseConstructor.construct_pairs
    с исправлением для того, чтобы предотвратить
    превращение ключей маппингов в числа или булевы значения. Мы хотим, чтобы
    они всегда были строками.
    WIKI-4860
    WIKI-6626
    """
    if not isinstance(node, yaml.MappingNode):
        raise yaml.constructor.ConstructorError(
            context=None,
            context_mark=None,
            problem='expected a mapping node, but found %s' % node.id,
            problem_mark=node.start_mark,
        )
    pairs = []
    for key_node, value_node in node.value:
        key = key_node.value
        value = loader.construct_object(value_node)
        pairs.append((key, value))
    return pairs


def build_form(source, **kwargs):
    if isinstance(source, str):
        declaration = parse_raw_declaration(source)
    else:
        declaration = source
    if declaration is None:
        raise FormSourceValidationError(
            code=FormSourceValidationError.CODES.YAML,
            message='Empty form given',
        )
    return build_form_from_declaration(declaration, **kwargs)


def parse_raw_declaration(source):
    data = load_yaml(source)
    fields = data and data.get('fields')
    if fields:
        data['fields'] = order_fields(fields, source)
    return data


def load_yaml(source):
    source = source.replace('options:', 'options: !OMap')
    return yaml.safe_load(source)


def order_fields(fields, source):
    ordered_fields = OrderedDict()
    ordered_keys = yaml_keys_natural_order(source, 'fields')
    for key in ordered_keys:
        if key in fields:
            # yaml_keys_natural_order finds too much
            # (not only form fields, also things that looks like
            # in template field for example)
            ordered_fields[key] = fields[key]
    return ordered_fields


def build_form_from_declaration(declaration, **kwargs):
    return ClearFormMapper.to_object(declaration, **kwargs)


class ClearFormMapper(object):
    """
    Создаем объект CleanForm из описания в виде yaml или дикта с описанием формы
    """

    @classmethod
    def to_object(
        cls, declaration, bind_fields=None, url=None, current_user=None, is_slave=False, files_fields=None, request=None
    ):
        """
        @param source: какой-нибудь текстовый источник Ямла или dict
        @param bind_fields: параметры передаваемые форме из POST
        """
        form_fields = OrderedDict()
        form_params = {}

        # this is a strange logic, at first
        # but bind_fields is either a manually constructed
        # dictionary or a QueryDict with property "_mutable"
        if bind_fields:
            if isinstance(bind_fields, dict) or not getattr(bind_fields, '_mutable', False):
                bind_fields = bind_fields.copy()

        for field, parameters in declaration.items():
            if field == 'fields':
                for field_name, field_options in parameters.items():
                    parsed = cls.parse_field(
                        name=field_name,
                        options=field_options,
                        bind_fields=bind_fields,
                    )
                    if parsed is not None:
                        form_fields[field_name] = parsed
            else:
                parsed = cls.parse_form_parameter(field, parameters)
                if parsed:
                    form_params.update({field: parsed})

        if url is not None:
            if not url.startswith('http'):
                url = '{0}://{1}/{2}'.format(settings.WIKI_PROTOCOL, settings.NGINX_HOST, url)

        target = declaration.get('target')
        if target is None and url is not None:
            form_params['target'] = url

        form_params['data'] = bind_fields
        if files_fields:
            form_params['files'] = files_fields

        form = ClearForm(**form_params)
        form.user = current_user
        form.fields = form_fields
        form.is_slave = is_slave
        form.request = request

        form.url = url
        return form

    @classmethod
    def parse_field(cls, name, options, bind_fields):
        if options is None:
            return

        for key in list(options.keys()):
            if not isinstance(key, str):
                # тут есть риск перезаписать
                # какой-нибудь существующий ключ
                # но он настолько мал, что невероятен.
                # Например, если есть маппинг:
                # field:
                #   True: 1
                #   true: 2
                # то field_options будут {'True': 1},
                # т.е. один ключ мы теряем.
                options[str(key)] = options[key]
                del options[key]

        type_ = options.get('type', 'string')
        options['type'] = type_

        obj = field_parsers.FIELD_HANDLERS[type_](name, **options)
        obj.label_top = bool(options.get('top'))
        obj.field_type = type_
        if 'params' in options:
            obj.params = options['params']

        if isinstance(obj, ClearStaffField):
            # hack ClearStaffField
            hidden_login = name + '__login'
            if bind_fields and bind_fields.get(hidden_login, None):
                bind_fields[name] = bind_fields[hidden_login]
                del bind_fields[hidden_login]
        elif isinstance(obj, ClearGroupField):
            hidden_url = name + '__url'
            if bind_fields and bind_fields.get(hidden_url, None):

                bind_fields[name] = bind_fields[hidden_url]
                del bind_fields[hidden_url]
        elif isinstance(obj, ClearServiceField):
            hidden_id = name + '__id'
            if bind_fields and bind_fields.get(hidden_id, None):

                bind_fields[name] = bind_fields[hidden_id]
                del bind_fields[hidden_id]
        return obj

    @classmethod
    def parse_form_parameter(cls, name, raw_value):
        """
        Параметры общие для формы
        """
        parameter_parsers = {
            'target': cls.parse_dummy,
            'title': cls.parse_dummy,
            'submit': cls.parse_string,
            'again': cls.parse_dummy,
            'success': cls.parse_string,
            'template': cls.parse_dummy,
            'html': cls.parse_dummy,
            'subject': cls.parse_dummy,
            'startrek_fields': cls.parse_dummy,
            'date_format': cls.parse_date_format,
        }
        if name in parameter_parsers:
            return parameter_parsers[name](raw_value)
        return

    @staticmethod
    def parse_dummy(value):
        return value

    @staticmethod
    def parse_string(value):
        return str(value)

    @staticmethod
    def parse_date_format(value):
        """
        формат даты (используется только в шаблоне для форматирования %date%).

        По умолчанию используется формат 2010-10-14.
        Вы должны использовать старый формат, аналогичный формату PHP-функции
        date (например, 'Y-m-d')
        """
        special_chars = 'aAwdbBmyYHIpMSfzZjUW'

        result = reduce(lambda a, b: a + ('%' if b in special_chars else '') + b, value, '')
        return result
