"""
Base class for all kind of form submit handlers
"""

import logging
import re
from datetime import datetime, time
from urllib.parse import urlparse

from django.utils.translation import ugettext, ugettext_lazy

from wiki.actions.classes.form_elements.fields import ALLOWED_STAFF_ATTRS
from wiki.intranet.models import Staff
from wiki.org import org_staff
from wiki.utils.supertag import translit
from wiki.utils.timezone import make_aware_utc, make_naive_msk, now

DEFAULT_TEMPLATE = '%fields%'

logger = logging.getLogger('form_submission')


def naive_moscow_time():
    """
    @type: datetime
    """
    return make_naive_msk(now())


def get_nice_staff_attribute(staff, attribute):
    def replace_family_status(replacement):
        replacement_map = {'S': ugettext_lazy('single'), 'M': ugettext_lazy('marriage')}
        return replacement_map.get(replacement, ugettext('not specified'))

    def replace_gender(replacement):
        replacement_map = {'F': ugettext_lazy('female'), 'M': ugettext_lazy('male')}
        return replacement_map.get(replacement, ugettext('not specified'))

    def replace_employment(replacement):
        replacement_map = {'F': ugettext('full'), 'P': ugettext('part-time'), 'D': ugettext('combining')}
        return replacement_map.get(replacement, ugettext('not specified'))

    def replace_domain(replacement):
        replacement_map = dict(Staff.DOMAIN_CHOICES)
        return replacement_map.get(replacement, ugettext('not specified'))

    replacement = getattr(staff, attribute, None)
    if replacement is None:
        replacement = ''
    elif not isinstance(replacement, str) and getattr(replacement, '__iter__', False):
        replacement = ', '.join([str(o) for o in replacement])
    elif type(replacement) == bool:
        if replacement:
            replacement = ugettext('yes')
        else:
            replacement = ugettext('no')
    else:
        replacement = str(replacement)
    attr_replace_strategy = {
        'family_status': replace_family_status,
        'gender': replace_gender,
        'employment': replace_employment,
        'domain': replace_domain,
    }.get(attribute, None)
    if attr_replace_strategy is not None:
        replacement = attr_replace_strategy(replacement)
    return replacement


class NoTitle(Exception):
    """
    Нужно вывести на печать в шаблон title, но title отсутствует.
    """

    pass


def process_title(title, param_value):
    """
    Вернуть title, если param_value == 'title', иначе вернуть param_value.

    @type title: str|None
    @type param_value: str
    @rtype: basestring
    """
    if param_value == 'title':
        if title is None:
            raise NoTitle
        return title
    else:
        return param_value


def process_params(form_params, title, true_or_false):
    """
    Обработать поле params как в WIKI-7036.

    @type form_params: dict
    @param form_params: {True: 'foo', False: 'bar'}
    @type true_or_false: bool
    @rtype: unicode
    """
    if true_or_false in form_params:
        # алиас для значения чекбокса все-таки задан в поле params
        try:
            return process_title(title, form_params[true_or_false])
        except NoTitle:
            return str(true_or_false)
    else:
        # не задан алиас для значения чекбокса
        return str(true_or_false)


class SubmitHandler(object):
    """
    Обрабатывает отправку формы
    """

    _subject = None

    field_wrapper = '<p>%s %s</p>'

    def __init__(self, form, **kwargs):
        self.form = form
        self.options = kwargs
        self.render_html = self.options.get('html', False)
        template = self.options.get('template')
        self.template = template or DEFAULT_TEMPLATE
        self.date_format = self.form.date_format
        self.staff_fields = {
            '%staff%': self.__fieldStaff,
            '%login%': self.__fieldLogin,
            '%position%': self.__fieldPosition,
            '%department%': self.__fieldDepartment,
            '%date%': self.__fieldDate,
            '%hiring_date%': self.__fieldHiringDate,
            '%user_groups%': self.__fieldStaffGroups,
            '%page%': self.__fieldUrl,
            '%boss%': self.__fieldBoss,
            '%department-boss%': self.__fieldDepartmentBoss,
        }

    @property
    def sender(self):
        """
        Сотрудник, который отправил форму.
        @rtype: intranet.Staff
        """
        return self.form.user.staff

    @property
    def sender_email(self):
        return self.sender.get_email()

    @property
    def page_url(self):
        return self.form.url

    @property
    def page_supertag(self):
        return translit(urlparse(self.page_url).path.strip('/'))

    def build_subject(self):
        """
        Сформировать тему для письма или summary для тикета

        @rtype: str
        """
        if self._subject is None:
            subject = self.options.get('subject')
            if not subject:
                form_title = self.form.form_title or ''
                self._subject = ugettext('A form was sent') + ' ' + form_title
            else:
                self._subject = self.process(template=subject)
            self._subject = self._subject.replace('\r', '').replace('\n', ' ')

        return self._subject

    def __fieldUrl(self):
        """
        Вернуть URL страницы с которой отправлены данные
        """
        return self.page_url

    def __fieldStaffGroups(self, delimiter=' / '):
        """
        Вернуть список подразделений, в которых состоит пользователь
        """
        groups = self.form.user.staff.all_groups
        return delimiter.join(map(str, groups))

    def __fieldHiringDate(self, formatted=True):
        """
        Вернуть дату приема на работу сотрудника в московском часовом поясе.
        """
        # join_at - DateField.
        dt = datetime.combine(self.form.user.staff.join_at, time())
        result = make_naive_msk(make_aware_utc(dt))
        if formatted:
            return result.strftime(self.date_format) + ' (MSK)'
        return result + ' (MSK)'

    def __fieldStaff(self):
        if hasattr(self, 'overrideStaff'):
            user = self.overrideStaff(self.form.user.staff)
        else:
            user = self.form.user.staff
        return str(user)

    def __fieldBoss(self):
        from wiki.actions.classes.form_elements.forms import get_boss

        boss = get_boss(self.form.user.staff)
        if boss is None:
            if getattr(self.form.user.staff, 'is_dismissed', False):
                return ugettext('not a member of any deparment')
            return ugettext('has no manager')
        if hasattr(self, 'overrideStaff'):
            boss = self.overrideStaff(boss)
        return str(boss)

    def __fieldDepartmentBoss(self):
        from wiki.actions.classes.form_elements.forms import get_department_boss

        boss = get_department_boss(self.form.user.staff)
        if boss is None:
            if getattr(self.form.user.staff, 'is_dismissed', False):
                return ugettext('not a member of any deparment')
            return ugettext('has no boss')
        if hasattr(self, 'overrideStaff'):
            boss = self.overrideStaff(boss)
        return str(boss)

    def __fieldLogin(self):
        return self.form.user.staff.login

    def __fieldPosition(self):
        return self.form.user.staff.position

    def __fieldDepartment(self):
        return self.form.user.staff.department.name

    def process(self, template=None):
        """
        Возвращает текстовое представление формы для сабмита
        """

        def in_fields_list(field_name, fields_list):
            for field in fields_list:
                if field_name == field['name']:
                    return field

        template = template or self.template
        result = template
        fields_list = self.form.for_handler()
        if '%fields%' in template:
            result = template.replace(
                '%fields%', '\n'.join([value for value in map(self.__process_field, fields_list) if value])
            )
        probable_keywords = re.compile(r'%[\w\d\-|]*%?').findall(template)
        probable_keywords.sort(key=lambda el: -len(el))
        for stop_word in probable_keywords:
            replacement = None
            field_infos = in_fields_list(stop_word[1:], fields_list)
            if field_infos:
                repl = self._override_field(field_infos)['value']
                replacement = str(repl)
            elif stop_word in self.staff_fields:
                replacement = self.staff_fields[stop_word]()
            elif stop_word[:7] == '%staff|' and stop_word[7:-1] in ALLOWED_STAFF_ATTRS:
                replacement = get_nice_staff_attribute(self.form.user.staff, stop_word[7:-1])
            elif any(stop_word[: len(macros)] == macros for macros in ('%boss|', '%department-boss|')):
                from wiki.actions.classes.form_elements.forms import get_boss, get_department_boss

                replacement_strategy = {'boss': get_boss, 'department-boss': get_department_boss}
                for macros in replacement_strategy.keys():
                    if stop_word.startswith('%' + macros):
                        got_macros = macros
                        break
                field_name = stop_word[len('%' + got_macros + '|') :]
                field_info = in_fields_list(field_name, fields_list)
                if field_info:
                    repl = self.form.cleaned_data.get(field_name)
                    try:
                        staff = org_staff().get(login=repl)
                    except (Staff.DoesNotExist, Staff.MultipleObjectsReturned):
                        pass
                    else:
                        if getattr(staff, 'is_dismissed', False):
                            replacement = ugettext('not a member of any department')
                        else:
                            boss = replacement_strategy[got_macros](staff)
                            if boss:
                                if hasattr(self, 'overrideStaff'):
                                    replacement = self.overrideStaff(boss)
                                else:
                                    replacement = str(boss)
                            else:
                                replacement = ugettext('form:HasNoDepartmentBoss')
            elif '%title%' == stop_word:
                replacement = self.form.form_title
            # do some actual job
            if replacement is not None:
                # call unicode() to make all lazy translations be evaluated
                result = result.replace(stop_word, str(replacement))
        return result

    def _substitute_placeholders(self, template):
        pass

    def __fieldDate(self):
        """
        Подставить значение текущей даты в Московском часовом поясе.
        """
        return str(naive_moscow_time().strftime(self.date_format)) + ' (MSK)'

    def _override_field(self, field_info):
        """
        Перехватить управление выводом на печать в классе наследнике

        @param field_info: словарь {имя_поля, поле, значение}
        """
        field = field_info['field']
        field_type = type(field).__name__[len('Clear') : -len('Field')].lower()

        override_method_name = '_override_' + field_type + '_field'
        override_method = getattr(self, override_method_name, lambda x: x)
        return override_method(field_info)

    def _get_cleaned_data_value(self, field_info):
        field_info['value'] = self.form.get_cleaned_data(field_info['name'])
        return field_info

    def _override_boolean_field(self, field_info):
        """
        Для поля типа чекбокс нужно учесть случай, когда пользователи хотят
        видеть в шаблоне имя поля (параметр title):

        Например,
        %%
        fields:
          ready_to_volunteer:
            type: checkbox
            title: Готов быть волонтером
            params:
              true: title
              false: "не готов быть волонтером"
        %%
        В параметрах params можно перечислить алиасы для "чекнутного" и "анчекнутого"
        чекбоксов. Для чекнутого используется слово "true". Его значение станет
        значением поля "ready_to_volunteer" в шаблоне, если чекбокс кликнут.
        Для анчекнутого используется слово "false". Если значение состоит
        из магического слова title, то в значение будет подставлено содержимое
        названия этого поля в вики-форме. См. пример: если чекнуть чекбокс
        и отправить форму, то в шаблон будет подсталена строка
        "Готов быть волонтером".
        """
        real_value = self.form.get_cleaned_data(field_info['name'])
        if hasattr(field_info['field'], 'params'):
            field_info['value'] = process_params(
                form_params=field_info['field'].params,
                title=getattr(field_info['field'], 'label', None),
                true_or_false=bool(real_value),
            )
        else:
            field_info['value'] = real_value
        logger.debug('Boolean field final value "%s". Was checkbox set?: "%s"', field_info['value'], real_value)
        return field_info

    _override_staff_field = _get_cleaned_data_value
    _override_group_field = _get_cleaned_data_value

    def __process_field(self, info):
        """
        Если попросили печатать все поля, применяем этот алгоритм

        @param info: {name, field, value}
        """
        field = self._override_field(info)
        if not field['value']:
            return None
        label = field['field'].label
        if self.render_html:
            return self.field_wrapper % (label if label else '', field['value'])
        return '%s %s' % (label + ':' if label else '', field['value'])

    def handle(self, target):
        for file in self.form.files.values():
            # нужно перемотать, потому что файл вычитывается в хендлерах.
            # TODO: после того как будет закопан старый фронтэнд:
            # можно будет не перематывать файл при сабмите формы на страницу или в excel-файл
            # перемотка означает вычитывание файла из хранилища, это долгая лишняя операция.
            file.seek(0)
