
import mimetypes
from email import encoders
from email.mime.base import MIMEBase

from django.core.mail.message import DEFAULT_ATTACHMENT_MIME_TYPE
from django.forms.fields import FileField
from django.utils.encoding import iri_to_uri

from wiki.actions.classes.form_elements.submit_handlers.base import SubmitHandler
from wiki.utils.mail import build_wiki_email_message


class EmailSubmitHandler(SubmitHandler):
    """
    Обрабатывает форму и отсылает письма
    """

    def _attach_files(self):
        attachments = []
        for name, field in self.form.fields.items():
            if isinstance(field, FileField):
                if self.form.get_cleaned_data(name, None) and name in self.form.files:
                    content = self.form.files[name].file.read()
                    mimetype = self.form.files[name].content_type
                    attachments.append(self._create_attachment(self.form.files[name].name, content, mimetype))
        return attachments

    def _create_mime_attachment(self, content, mimetype):
        """
        Калька с lib.email.mime.base
        """
        basetype, subtype = mimetype.split('/', 1)
        attachment = MIMEBase(basetype, subtype)
        attachment.set_payload(content)
        encoders.encode_base64(attachment)
        return attachment

    def _create_attachment(self, filename, content, mimetype=None):
        """
        Это калька с email.mime.base, единственное отличие --
        заголовок Content-Disposition, поддерживающий UTF

        Если есть более простой способ отправлять имена файлов в UTF, не
        используя этот метод, то используйте этот этот метод.
        """
        if mimetype is None:
            mimetype, _ = mimetypes.guess_type(filename)
            if mimetype is None:
                mimetype = DEFAULT_ATTACHMENT_MIME_TYPE
        attachment = self._create_mime_attachment(content, mimetype)

        if filename:
            if all(ord(c) < 128 for c in filename):
                params = {'filename': filename}
            else:
                params = {'filename*': "UTF-8''%s" % iri_to_uri(filename)}
            attachment.add_header('Content-Disposition', 'attachment', **params)
        return attachment

    def _override_char_field(self, field_info):
        if self.render_html:
            value = field_info['value']
            field_info['value'] = value.replace('\n', '\n<br />')
        return field_info

    def handle(self, target):
        author_name = self.sender.get_full_name()
        from_email = self.sender_email

        to = target.to
        cc = target.params.get('cc')
        bcc = target.params.get('bcc')
        body = self.process()

        message = build_wiki_email_message(
            to=to,
            cc=cc,
            bcc=bcc,
            from_email=from_email,
            reply_to=from_email,
            author_name=author_name,
            subject=self.build_subject(),
            body=body,
            attachments=self._attach_files(),
            supertag=self.page_supertag,
            content_subtype='html' if self.render_html else 'plain',
            # WIKI-7026 — email from form shouldn't have References header
            add_references_header=False,
        )

        message.send()
        super(EmailSubmitHandler, self).handle(target)
