
import datetime
import urllib.parse

from django.forms.fields import FileField
from django.utils.encoding import smart_text
from django.utils.translation import ugettext

from wiki.actions.classes.form_elements.form_errors import FormHandlingError
from wiki.actions.classes.form_elements.submit_handlers.base import SubmitHandler
from wiki.files.api import upload_file
from wiki.intranet.models import Staff
from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.org import get_org
from wiki.pages.api import save_page
from wiki.pages.models import Page, Revision
from wiki.pages.models.page import RedirectLoopException
from wiki.utils.diff import simple_diff
from wiki.utils.supertag import translit
from wiki.utils.timezone import now


class PageAnchorSubmitHandler(SubmitHandler):
    """
    Вставка данных формы в страницу по якорю
    """

    def __init__(self, form, **kwargs):
        super(PageAnchorSubmitHandler, self).__init__(form, **kwargs)
        self.all_files = {}

    def _override_staff_field(self, field_info):
        field = field_info['field']
        if not field_info['value'] or not field.show:
            name = field_info['name']
            people = []
            for ya in field.get_staff(self.form.cleaned_data.get(name)):
                if isinstance(ya, Staff):
                    people.append('staff:' + ya.login)
                else:
                    people.append(ya)
            field_info['value'] = ', '.join(people)
        return field_info

    def overrideStaff(self, staff):
        return 'staff:' + str(staff.login)

    def handle(self, target):
        """
        override option for as_html rendering, because no html in
        page.body field is needed - it uses it's own special markup
        """
        self.render_html = False
        pages = list()
        if isinstance(target.to, urllib.parse.ParseResult):
            urls = [target.to]
        else:
            urls = target.to

        for url in urls:
            try:
                pages.append(Page.objects.get(supertag=translit(url.path), org=get_org()))
            except Page.DoesNotExist:
                raise FormHandlingError(
                    FormHandlingError.CODES.MISSING_PAGE,
                    arg=url.path,
                    message=(
                        ugettext("Form is trying to submit to page that doesn't exist")
                        + ': '
                        + smart_text(url.path, strings_only=True)
                    ),
                )
        result = {}
        for page in pages:
            try:
                target_page = page.redirect_target()
            except RedirectLoopException:
                target_page = page
            result[target_page] = []
            for url in urls:
                if page.supertag == translit(url.path):
                    result[target_page].append(url.fragment)
        self._attach_files_to_pages(list(result.keys()))
        # this call MUST be after self._attach_files_to_pages, sorry!
        processed_form = self.process()

        for page, fragments in result.items():
            before = page.body
            for anchor in fragments:
                body = new_wf_format(page, anchor, '\n' + processed_form)
                save_page(page, body, request=self.form.request, create_revision=False)

            revision = Revision.objects.create_from_page(page)
            diff = simple_diff(before, page.body)
            dummy = {}
            if diff[0]:
                dummy['added'] = '<div class="additions">' + '\n'.join(diff[0]) + '</div>'
            if diff[1]:
                dummy['deleted'] = '<div class="deletions">' + '\n'.join(diff[1]) + '</div>'
            dummy['revision_id'] = revision.id
            PageEvent(
                timeout=now() + datetime.timedelta(minutes=5),
                author=self.form.request.user,
                page=page,
                meta=dummy,
                event_type=EventTypes.edit,
                notify=False,
            ).save()
            page.save()
        super(PageAnchorSubmitHandler, self).handle(target)

    def _attach_files_to_pages(self, pages):
        """
        Если у формы есть файлы, то прицепить все эти файлы к страницам

        К каждой странице цепляется свой файл,
        т.к. отношение Файл-Страница = many-to-one
        """
        for name, field in self.form.fields.items():
            if isinstance(field, FileField):
                if self.form.get_cleaned_data(name, None) and name in self.form.files:
                    for page in pages:
                        username = str(self.form.user.staff)

                        self.all_files[name] = upload_file(
                            self.form.files[name],
                            page,
                            self.form.user,
                            '%s от %s' % (self.form.fields[name].label, username),  # @todo: need l10n
                        )


def new_wf_format(page, anchor, insert):
    anchor_actions = list(page.wiki_elem('action', name=('anchor', 'a')))
    body = page.body

    matched_anchor_found = False
    for action in reversed(anchor_actions):
        params = action._children
        if not params:
            continue
        anchor_name = params[0].value
        if anchor_name == anchor:
            body = body[: action.pos_end] + insert + body[action.pos_end :]
            matched_anchor_found = True

    # если якорей нет, либо ни один из них не совпадает с искомым, дописываем в конец
    if not anchor_actions or not matched_anchor_found:
        body += insert

    return body
