
import io
import os
from urllib.parse import urlparse

import xlrd
import xlwt
from django.core.files.uploadedfile import InMemoryUploadedFile
from django.db import transaction
from django.db.models import Q
from django.forms.fields import FileField
from django.utils.translation import ugettext

from wiki.actions.classes.form_elements.form_errors import FormHandlingError
from wiki.actions.classes.form_elements.submit_handlers.base import SubmitHandler
from wiki.files import logic as files_logic
from wiki.files.api import upload_file
from wiki.files.models import File
from wiki.org import get_org
from wiki.pages.models import Page
from wiki.utils import smart_list
from wiki.utils.supertag import translit


class PageFileSubmitter(SubmitHandler):
    """
    Дописать в Excel файл данные из формы
    """

    def xls(self):
        """
        Предполагаем, что вы в шаблоне указали поля формы, разделенные ";"
        """
        wb = xlwt.Workbook()
        wb.add_sheet('Данные формы' + (' ' + self.form.form_title if self.form.form_title else ''))
        return wb

    def process_sheet(self, sheet, at, form):
        for index in range(len(form)):
            sheet.write(at, index, form[index])

    def _attach_files_to_pages(self, pages):
        """
        Если у формы есть файлы, то прицепить все эти файлы к страницам

        К каждой странице цепляется свой файл,
        т.к. отношение Файл-Страница = many-to-one
        """
        for name, field in self.form.fields.items():
            if isinstance(field, FileField):
                if self.form.get_cleaned_data(name, None) and name in self.form.files:
                    for page in pages:
                        username = str(self.form.user.staff)

                        self.all_files[name] = upload_file(
                            self.form.files[name],
                            page,
                            self.form.user,
                            '%s от %s' % (self.form.fields[name].label, username),  # @todo: need l10n
                        )

    def _form_as_list(self):
        """
        Возвращает представление формы в виде списка
        """
        result = []
        for template in self.template.split(';'):
            result.append(self.process(template))
        return result

    @staticmethod
    def split_target(target):
        xlsx_ext = '.xlsx'
        if target.endswith(xlsx_ext):
            target = target[: -len(xlsx_ext)] + '.xls'
        elif not target.endswith('.xls'):
            target += '.xls'
        separator = target.rfind('/')
        if separator > -1:
            supertag, filename = target[:separator], target[separator + 1 :]
            supertag = translit(supertag)
            if not filename:
                filename = ugettext('FormConstructor')
        else:
            filename = target
            supertag = None
        return filename, supertag

    def handle(self, target):
        purl = urlparse(self.form.url)
        page = Page.objects.get(supertag=translit(purl.path[1:]), org=get_org())

        pages_filenames = {}
        targets = smart_list(target.to)
        for target in targets:
            filename, supertag = self.split_target(target)
            if supertag is not None:
                try:
                    target_page = Page.objects.get(supertag=supertag, org=get_org())
                except Page.DoesNotExist:
                    raise FormHandlingError(
                        FormHandlingError.CODES.MISSING_PAGE,
                        arg=target,
                        message=ugettext("Form is trying to submit to page that doesn't exist") + ': ' + target,
                    )
            else:
                target_page = page
            target_page = target_page.redirect_target()
            if target_page not in pages_filenames:
                pages_filenames[target_page] = []
            pages_filenames[target_page].append(filename)

        processed_form = self._form_as_list()
        for page, filenames in pages_filenames.items():

            files = File.active.filter(page=page).filter(Q(url__in=filenames) | Q(name__in=filenames))

            for file_name in filenames:
                requested_files = []
                for file_model in files:
                    if file_model.name == file_name:
                        requested_files.append(file_model)
                        break

                write_at_line = 0
                if not requested_files:
                    # create workbook
                    wb = self.xls()
                else:
                    file_model = requested_files[0]
                    old_wb = xlrd.open_workbook(file_contents=file_model.mds_storage_id.read())
                    wb, write_at_line = self.copy_workbook(old_wb)
                sheet = wb.get_sheet(0)
                self.process_sheet(sheet, write_at_line, processed_form)
                xls = io.BytesIO()
                wb.save(xls)
                xls.seek(0, os.SEEK_END)
                size = xls.tell()
                xls.seek(0)

                uploaded_file = InMemoryUploadedFile(
                    file=xls,
                    field_name='file',
                    name=file_name,
                    content_type='application/msexcel',
                    size=size,
                    charset='',
                )

                with transaction.atomic():
                    if requested_files:
                        files_logic.delete(
                            file=file_model,
                            user=self.form.user,
                        )
                    upload_file(uploaded_file, page, self.form.user, check_access=False)

        self._attach_files_to_pages([page])
        super(PageFileSubmitter, self).handle(target)

    def copy_workbook(self, wb):
        wwb = xlwt.Workbook()
        sheet = wwb.add_sheet(ugettext('Form data') + (' ' + self.form.form_title if self.form.form_title else ''))
        for x in range(10000):
            try:
                for y in range(100):
                    value = wb.sheet_by_index(0).cell(x, y).value
                    if value:
                        sheet.write(x, y, value)
            except IndexError:
                if y == 0:
                    break
                else:
                    x += 1
        return wwb, x
