
import itertools
from operator import attrgetter
from urllib.parse import ParseResult


class Target(object):
    def __init__(self, type, to, **params):
        self.type = type
        self.to = to
        self.params = params
        super(Target, self).__init__()

    def __str__(self):
        representation = str(self.to)
        if self.params:
            representation += '?' + '&'.join('%s=%s' % param_pair for param_pair in list(self.params.items()))
        return representation

    def __repr__(self):
        return '<Target: %s>' % str(self)


def merge_targets(targets):
    """
    Смержить несколько таргетов в один.
    Мержить можно таргеты одного типа.
    Пока мерженье заключается в умном мерженье списков с
    несписками в атрибуте `to`.

    @rtype: Target
    """
    # проверка одинаковости типов
    targets_types = set(t.type for t in targets)
    assert len(targets_types) == 1

    common_type = targets_types.pop()

    to_lists = []
    for target in targets:
        if isinstance(target.to, (str, ParseResult)):
            to_list = [target.to]
        else:
            to_list = target.to
        to_lists.append(to_list)

    return Target(type=common_type, to=list(itertools.chain.from_iterable(to_lists)))


def group_targets(targets, grouper=attrgetter('type')):
    """
    Разобрать таргеты по группам.
    Возвращает пары группировочный параметр, список сгруппированных таргетов.
    """
    for type, group in itertools.groupby(sorted(targets, key=grouper), grouper):
        yield type, list(group)
