
from django.forms import CharField, Form
from django.utils.translation import ugettext

from .base_widget_action import WikiBaseAction, WikiWidgetAction
from .params_validator import BooleanFromTextField
from wiki.api_frontend.views.grids import get_grid_data
from wiki.grids.models import Grid as GridModel
from wiki.org import get_org
from wiki.pages.logic import url_to_wiki_page_in_frontend
from wiki.pages.models import Page, RedirectLoopException


class Grid(WikiWidgetAction):
    """
    Action {{grid}} lets insert "table-lists" (aka "grids") on wiki pages.
    The only mandatory parameter is page="<table-list address>".
    """

    default_params = {'readonly': False, 'filter': None, 'columns': None, 'num': True, 'width': None, 'sort': True}

    @staticmethod
    def _parse_params(params):
        result = params.dict
        if 'page' in result:
            pass
        elif 'url' in result:
            result['page'] = result['url']
            del result['url']
        else:
            first_ordered_param = next(WikiBaseAction.ordered_params(params), None)
            if first_ordered_param is not None:
                result['page'] = first_ordered_param

        return result

    def encode_params(self, params):
        # URL грида можно задать тремя способами:
        # 1) Через параметр page, например {{grid page=/foo/bar}}
        # 2) Через параметр url, например {{grid url=/foo/bar}}
        # 3) Через первый ordered_param, например {{grid /foo/bar}}
        # Приоритет в определении итогового URL идет
        # в порядке номеров способов: 1)-2)-3)
        result = self._parse_params(params)

        # Указанная в параметре page страница может быть редиректом.
        if 'page' in result:
            page = None
            try:
                page = Page.active.get(supertag=self.normalized_supertag(result['page']), org=get_org())
                page = page.redirect_target()
            except Page.DoesNotExist:
                self.error_happened(ugettext('actions.Grid:NonexistentGridPage'))
            except RedirectLoopException:
                pass
            result['page'] = page.supertag

        return result

    def json_for_get(self, params):
        grid = None
        try:
            grid = GridModel.objects.get(supertag=params.get('page'), org=get_org())
        except GridModel.DoesNotExist:
            self.error_happened(ugettext('actions.Grid:Not a grid'))

        if not grid.has_access(self.request.user):
            self.error_happened(
                ugettext('actions.Grid:AccessDenied {url}').format(url=url_to_wiki_page_in_frontend(grid))
            )

        return get_grid_data(grid, self.request)

    def form_class_to_validate_params(self):
        class Validator(Form):
            page = CharField(required=True)
            readonly = BooleanFromTextField(required=False)
            filter = CharField(required=False)
            columns = CharField(required=False)
            num = BooleanFromTextField(required=False)
            width = CharField(required=False)
            sort = BooleanFromTextField(required=False)

        return Validator

    def get_root_page_tag_from_params(self):
        """
        Вернуть ссылку на таблицу, указанную в вики тексте действия.
        """
        return self._parse_params(self.params).get('page')
