"""
    This module provides action {{mypages}}
"""
import json
from datetime import date

from django.utils.translation import ugettext

from wiki.actions.classes.base_action_deprecated import OldDjangoWidgetAction
from wiki.org import get_org, org_staff
from wiki.pages.access import ACCESS_DENIED, get_bulk_access_status, is_admin
from wiki.pages.models import Page

by_time = (lambda page: page.modified_at.date(),)

by_created_at = (lambda page: page.created_at.date(),)

by_letter = (lambda page: page.supertag[0],)


def qs_splitter_by_value(qs, unit_getter, equality_measure=None):
    if equality_measure is None:
        equality_measure = lambda current_value, value: current_value == value
    result = []
    cur_value = None
    container = []
    for entity in qs:
        if cur_value is None:
            cur_value = unit_getter(entity)
            result.append((cur_value, container))
        else:
            unit = unit_getter(entity)
            if not equality_measure(cur_value, unit):
                cur_value = unit
                container = []
                result.append((unit, container))
        container.append(entity)
    return result


class Mypages(OldDjangoWidgetAction):
    """
    Action {{mypages}} shows all pages by the user viewing current page,
    or by any arbitrary user
    """

    @property
    def is_json_rendering(self):
        return self.params.get('render_json') is not None

    @is_json_rendering.setter
    def is_json_rendering(self, value):
        # игнорируем setter, поскольку значение зависит только от GET параметра 'render_json'
        pass

    def render_json(self):
        context_data = self.get_context_data()
        if 'split_entities' in context_data:
            split_entities = context_data['split_entities']
            context_data['split_entities'] = [
                {
                    'key': key.strftime('%Y-%m-%d %H:%M:%S') if isinstance(key, date) else key,
                    'pages': [
                        {
                            'tag': entity.tag,
                            'url': entity.url,
                            'created_at': entity.created_at.strftime('%Y-%m-%d %H:%M:%S'),
                            'modified_at': entity.modified_at.strftime('%Y-%m-%d %H:%M:%S'),
                        }
                        for entity in entities
                    ],
                }
                for key, entities in split_entities
            ]
        return json.dumps(context_data)

    def get_context_data(self):
        """
        Return context for template mypages.html
        (Callback for base class's render())
        """
        invalid_login = False
        accessible_pages = []

        login = self.params.get('user') or self.params.get('login') or self.params.get('owner')

        if not login:
            login = self.user.username
        else:
            staff_set = org_staff().filter(user__username=login)
            if not staff_set.exists():
                error_msg = ugettext('actions.mypages:UnknownUser') + ' ' + login
                if self.is_json_rendering:
                    self.http_status_code = 404
                    return {
                        'error': error_msg,
                    }

                self.add_error(error_msg)
                invalid_login = True

        sort_by_change = self.is_param('bychange') or self.request.GET.get('bychange')
        sort_by_date = self.is_param('bydate') or self.request.GET.get('bydate')
        sort_by_letter = not sort_by_change and not sort_by_date
        sort_by_fields = []
        if sort_by_change:
            sort_by_fields.append('modified_at')
        if sort_by_date:
            sort_by_fields.append('created_at')
        if not sort_by_fields:
            sort_by_fields.append('supertag')

        if not invalid_login:
            pages = (
                Page.active.filter(authors__username=login, org=get_org())
                .order_by(*sort_by_fields)
                .only('tag', 'supertag', 'modified_at', 'created_at', 'title')[:500]
            )
            supertags = [page.supertag for page in pages]
            if is_admin(self.request.user):
                accessible_pages = pages
            else:
                access_rules = get_bulk_access_status(supertags, self.request.user)
                accessible_pages = []
                for page in pages:
                    access_level = access_rules.get(page.supertag, ACCESS_DENIED)
                    if access_level > ACCESS_DENIED:
                        accessible_pages.append(page)
        if sort_by_change:
            order_name = ugettext('by modification date')
            args = by_time
        elif sort_by_date:
            order_name = ugettext('by creation date')
            args = by_created_at
        else:
            order_name = ugettext('alphabetically')
            args = by_letter
        return {
            'login': login,
            'sort_by_change': sort_by_change,
            'sort_by_date': sort_by_date,
            'sort_by_letter': sort_by_letter,
            'order_by': order_name,
            'split_entities': qs_splitter_by_value(accessible_pages, *args),
        }
