
from itertools import groupby
from urllib.parse import urlparse

from django.conf import settings
from django.utils.translation import ugettext
from ids.exceptions import IDSException

from .base_widget_action import WikiWidgetAction
from wiki.utils import planner

HOST_BY_CONTACT_VALIDATOR = {
    'URL': '',
    'WIKI': f'https://{settings.NGINX_HOST}/',
    'MAILLIST': 'https://ml.yandex-team.ru/',
    'AT_CLUB': 'https://clubs.at.yandex-team.ru/',
    'STARTREK': 'https://st.yandex-team.ru/',
    'ST_FILTER': '',
    'METRIKA_ID': 'http://h.yandex.net/?http%3A//metrika.yandex.ru/stat/dashboard/%3Fcounter_id%3D',
}


class ServiceNotFound(Exception):
    pass


class ServiceParamNotSupported(Exception):
    pass


class InvalidIdParam(Exception):
    pass


class ManyServicesFound(Exception):
    def __init__(self, supertag, matched_services):
        self.supertag = supertag
        self.matched_services = matched_services


class Serviceteam(WikiWidgetAction):
    services_repo = planner.get_planner_services_repository()
    service_members_repo = planner.get_planner_service_members_repository()
    service_contacts_repo = planner.get_planner_service_contacts_repository()
    parsed_planner_url = urlparse(settings.PLANNER_URL)

    def get_search_params(self, params):
        """
        Сформировать и вернуть словарь с параметрами для поиска сервиса.

        @type params: dict
        @rtype dict
        """
        query_builders = (
            self._query_by_id_param,
            self._query_by_slug_param,
            self._query_by_name_param,
            self._query_by_service_param,
            self._query_by_supertag,
        )

        for builder in query_builders:
            search_params = builder(params)
            if search_params:
                return search_params

        raise ServiceNotFound()

    def _query_by_id_param(self, params):
        id_param = params.get('id')
        if id_param is None:
            return

        try:
            service_id = int(id_param)
        except ValueError:
            raise InvalidIdParam()
        else:
            return {'id': service_id}

    def _query_by_slug_param(self, params):
        slug_param = params.get('slug')
        if slug_param is None:
            return

        return {'slug': slug_param}

    def _query_by_name_param(self, params):
        name_param = params.get('name')
        if name_param is None:
            return

        try:
            name_param.encode('ascii')
        except UnicodeError:
            # если значение не из латинских букв,
            # ищем по названию сервиса на русском
            attr_name = 'name__contains'
        else:
            # если значение из латинских букв,
            # ищем по названию сервиса на английском
            attr_name = 'name_en__contains'

        return {attr_name: name_param}

    def _query_by_service_param(self, params):
        service_param = params.get('service')
        if service_param is None:
            return

        # параметр service больше не поддерживается
        raise ServiceParamNotSupported()

    def _query_by_supertag(self, params):
        supertag = self.page.supertag
        tag = self.page.tag

        lookup = {'contact_type_code': 'url_wiki', 'content__contains': supertag, 'fields': 'service.id, service.name'}
        service_contacts = self.service_contacts_repo.getiter(lookup)

        if not service_contacts.total:
            lookup['content__contains'] = tag
            # После реализации https://st.yandex-team.ru/ABC-5385 можно будет объединить два условия (выше и ниже)
            # через OR
            service_contacts = self.service_contacts_repo.getiter(lookup)

        if not service_contacts.total:
            return

        first_page = service_contacts.first_page
        # список контактов - это список сущностей ServiceContact из разных сервисов, содержащий внутри сущности Service
        # как признак принадлежности к конкретному сервису
        unique_service_ids = set([contact['service']['id'] for contact in first_page])
        if len(unique_service_ids) == 1:
            return {'id': unique_service_ids.pop()}
        else:
            raise ManyServicesFound(
                supertag=supertag,
                matched_services=list(
                    {contact['service']['id']: contact['service'] for contact in first_page}.values()
                ),
            )

    def _get_contacts(self, service):
        contacts = []
        service_contacts = self.service_contacts_repo.get(
            {'service': service['id'], 'fields': 'type.code,type.validator,content,title'}
        )
        for service_contact in service_contacts:
            type = service_contact['type']['code']
            validator = service_contact['type']['validator']
            content = service_contact['content']
            if not content.startswith('http'):
                content = '{}{}'.format(HOST_BY_CONTACT_VALIDATOR.get(validator, ''), content)
            title = service_contact['title']
            # WIKI-15463: заменяем отсутствующие переводы на значения для языка 'ru'
            title = {key: choose_for_language(title, key) for key in title.keys()}
            contacts.append(
                {
                    'type': type,
                    'link': content,
                    'text': title,
                }
            )

        return contacts

    def _get_roles(self, service):
        members = self.service_members_repo.get(
            {'service': service['id'], 'fields': 'role.name,role.id,person.login,person.name'}
        )

        get_role_id = lambda m: m['role']['id']
        get_role = lambda m: m['role']

        member_sorted_by_role = sorted(members, key=get_role_id)
        members_grouped_by_role = groupby(member_sorted_by_role, get_role)

        roles = []
        for role, role_members in members_grouped_by_role:
            role_name = choose_for_language(
                obj=role['name'],
                lang=self.request.LANGUAGE_CODE,
            )
            persons = [member['person'] for member in role_members]

            roles.append(
                {
                    'name': role_name,
                    'persons': persons,
                }
            )

        return roles

    def json_for_get(self, params):
        try:
            search_params = self.get_search_params(params)
            search_params['fields'] = 'id,name'
        except ServiceNotFound:
            return self.report_service_not_found(params)
        except InvalidIdParam:
            return self.report_invalid_id_param()
        except ServiceParamNotSupported:
            return self.report_service_param_not_supported()
        except ManyServicesFound as exc:
            return self.report_many_services(exc)

        try:
            service = self.services_repo.get_one(search_params)
        except IDSException:
            return self.report_service_not_found(params)

        data = {}
        if 'contacts' in params:
            data['contacts'] = self._get_contacts(service)
        else:
            data['roles'] = self._get_roles(service)

        team_path = settings.SERVICETEAM_EDIT_URL.format(id=service['id'])
        data['edit_url'] = self.parsed_planner_url._replace(path=team_path).geturl()
        data['service_name'] = choose_for_language(
            obj=service['name'],
            lang=self.request.LANGUAGE_CODE,
        )

        return data

    def report_many_services(self, exc):

        # Translators:
        #   ru: Под именем проекта {projectcode} известно несколько сервисов.
        #   Укажите вместо {projectcode} числовой идентификатор:
        message = ugettext('actions.Serviceteam:FoundManyServices {projectcode}')

        # Translators:
        #   ru: Для сервиса «{servicename}» укажите параметр id={serviceid}
        id_message = ugettext('actions.Serviceteam:UseServiceId {servicename} {serviceid}')

        message_parts = [message.format(projectcode=exc.supertag)]
        for service in exc.matched_services:
            message_parts.append(
                id_message.format(
                    servicename=choose_for_language(obj=service['name'], lang=self.request.LANGUAGE_CODE),
                    serviceid=service['id'],
                )
            )
        self.error_happened(', '.join(message_parts))

    def report_service_not_found(self, params):
        # Имена параметров указаны в том же порядке, в котором
        # они проверяются при построении поискового запроса.
        # Это важно для корректного отображения ошибки
        # 'actions.Serviceteam:NoSuchServiceByParam {paramname}'.
        search_param_names = ('id', 'slug', 'name')

        if any(param in params for param in search_param_names):
            # Translators:
            #   ru: Сервис, указанный в параметре {paramname}, не найден.
            message = ugettext('actions.Serviceteam:NoSuchServiceByParam {paramname}')
            param_name = next(param for param in search_param_names if param in params)
            message = message.format(paramname=param_name)
        else:
            # Translators:
            #   ru: Не могу однозначно определить сервис для текущей
            #   страницы. Укажите название или ID сервиса в параметре id=''
            message = ugettext('actions.Serviceteam:CannotGuessService')

        self.error_happened(message)

    def report_invalid_id_param(self):
        # Translators:
        #   ru: Для поиска сервиса по имени или slug вместо параметра id
        #   используйте параметры name или slug, соответсвенно.
        #   en: To search for the service by name or slug use the 'name' or 'slug' parameter
        #   respectively instead of the 'id' parameter.
        message = ugettext('actions.Serviceteam:InvalidIdParam')
        self.error_happened(message)

    def report_service_param_not_supported(self):
        # Translators:
        #   ru: Параметр service в экшене больше не поддерживается. Пожалуйста, замените его на параметр name.
        #   en: The 'service' parameter is no longer supported. Please replace it with the 'name' parameter.
        message = ugettext('actions.Serviceteam:ServiceParamNotSupported')
        self.error_happened(message)


def choose_for_language(obj, lang):
    """
    Берем дикт с полями 'ru'/'en'/может еще какими-нибудь и берем из него
    значение в соответствии языком `lang`, если оно непустое.
    Если пустое — фолбечимся на 'ru'.
    """

    if lang != 'ru' and obj[lang]:
        return obj[lang]
    return obj['ru']
