
from django.conf import settings
from django.forms import Form, IntegerField, ChoiceField, CharField

from wiki.api_core.utils import get_nginx_host
from wiki.pages.pages_tree_new import PagesTree, ExpandSubtreeUrlBuilder, SortBy, Sort
from wiki.utils.supertag import translit

from .base_widget_action import WikiWidgetAction
from .params_validator import BooleanFromTextField


class Tree(WikiWidgetAction):
    """
    Экшен дерева страниц.

    https://wiki.yandex-team.ru/wiki/components/pages-tree/
    """

    default_params = {
        'depth': 3,
        'show_redirects': False,
        'show_grids': True,
        'show_files': False,
        'show_owners': False,
        'show_titles': True,
        'show_created_at': False,
        'show_modified_at': False,
        'sort_by': SortBy.DEFAULT,
        'sort': Sort.DEFAULT,
        'nomark': False,
        'authors': '',
    }

    def get_root_page_tag_from_params(self):
        """
        Вернуть ссылку на страницу, указанную в вики тексте действия.
        """
        return self.params.dict.get('page') or self.params.dict.get('for')

    def encode_params(self, params):
        params.dict['page'] = self._determine_root_page_supertag(params)
        params.dict['nomark'] = self._determine_nomark(params)
        return params.dict

    def json_for_get(self, params):
        expand_subtree_url_builder = ExpandSubtreeUrlBuilder(
            get_params=params,
            nginx_host=get_nginx_host(self.request),
        )
        to_bool = BooleanFromTextField.to_bool

        try:
            depth = int(params['depth'])
        except ValueError:
            depth = None

        return PagesTree(
            root_supertag=translit(params['page']),
            user=self.request.user,
            expand_subtree_url_builder=expand_subtree_url_builder,
            depth=depth or None,
            show_redirects=to_bool(params['show_redirects']),
            show_grids=to_bool(params['show_grids']),
            show_files=to_bool(params['show_files']),
            show_owners=to_bool(params['show_owners']),
            show_titles=to_bool(params['show_titles']),
            show_created_at=to_bool(params['show_created_at']),
            show_modified_at=to_bool(params['show_modified_at']),
            sort_by=params['sort_by'],
            sort=params['sort'],
            from_yandex_server=self.request.from_yandex_server,
            authors=params['authors'].split(',') if params['authors'] else [],
        ).data

    def form_class_to_validate_params(self):
        # django.forms.ChoiceField требует кортеж
        # из двух значений на каждую опцию.
        sort_by_choices = [(x, x) for x in SortBy.CHOICES]
        sort_choices = [(x, x) for x in Sort.CHOICES]

        class Validator(Form):
            depth = IntegerField(required=False, min_value=0)
            show_redirects = BooleanFromTextField(required=False)
            show_grids = BooleanFromTextField(required=False)
            show_files = BooleanFromTextField(required=False)
            show_owners = BooleanFromTextField(required=False)
            show_titles = BooleanFromTextField(required=False)
            show_created_at = BooleanFromTextField(required=False)
            show_modified_at = BooleanFromTextField(required=False)
            sort_by = ChoiceField(choices=sort_by_choices, required=False)
            sort = ChoiceField(choices=sort_choices, required=False)
            nomark = BooleanFromTextField(required=False)
            subtree = BooleanFromTextField(required=False)
            authors = CharField(required=False)

        return Validator

    def _determine_root_page_supertag(self, params):
        # Корневую страницу дерева можно задавать параметрами 'for' или 'page',
        # если эти параметры не заданы, то дерево строится для текущей страницы.

        root_page_path = params.dict.get('page') or params.dict.get('for')

        if 'for' in params.dict:
            del params.dict['for']

        if not root_page_path:
            if self.request.supertag:
                # Уже определена какая-то страница
                return self.request.supertag
            else:
                # Экшн вызван у корня, для совместимости считаем, что это домашняя страница
                return settings.MAIN_PAGE
        else:
            root_page_path = root_page_path.strip()

            # homepage alias
            if root_page_path == '/':
                return settings.MAIN_PAGE

        return self.normalized_supertag(root_page_path)

    def _determine_nomark(self, params):
        # Параметр nomark имеет синоним subtree

        nomark = params.dict.get('nomark') or params.dict.get('subtree')

        if 'subtree' in params.dict:
            del params.dict['subtree']

        return nomark
