import logging

from django.conf import settings
from django.http import HttpResponseForbidden, HttpResponseNotFound

from wiki.api.views.api_view import ApiView
from wiki.api_core.utils import is_tvm_authentication
from wiki.files.api import response_to_download_file
from wiki.files.models import File
from wiki.org import get_org
from wiki.pages.access import has_access
from wiki.pages.dao.page import get_page_by_supertag
from wiki.pages.models import Page, RedirectLoopException
from wiki.utils.supertag import translit

logger = logging.getLogger('django.request')


class DownloadView(ApiView):
    """
    Обрабатывает запросы от бэкенда Доквьювера на загрузку Вики файлов для просмотра.
    Требует аутентификацию по TVM2 service и user тикетам.
    """

    TVM2_AUTH = True

    def dispatch(self, request, *args, **kwargs):
        # проверим, что к нам пришел бэкенд Доквьювера
        allowed_clients = (settings.DOCVIEWER_TVM2_CLIENT_ID, settings.YAUTH_TVM2_CLIENT_ID)
        if is_tvm_authentication(self.request) and str(self.request.yauser.service_ticket.src) not in allowed_clients:
            return HttpResponseForbidden(
                'TVM client with id={} has no access'.format(self.request.yauser.service_ticket.src)
            )

        if request.user.is_anonymous:
            return HttpResponseForbidden('User is not authenticated')

        # для B2B проверить организацию (она должна определяться по tvm user ticker в org_detector middleware)
        if settings.IS_BUSINESS and request.org is None:
            return HttpResponseForbidden('Unknown organization')

        return super(DownloadView, self).dispatch(request, *args, **kwargs)

    def get(self, request):
        fileid = request.GET.get('fileid', '')
        force_download = str(request.GET.get('download', '0')) == '1'
        # выкусываем из fileid значение org_id (fileid=page_supertag/file_url?org_id)
        fileid = fileid.split('?')[0]
        separator = fileid.rfind('/')

        logger.info('Trying to download {} file'.format(fileid))

        supertag, url = fileid[:separator], fileid[separator + 1 :]
        url = translit(url)
        try:
            page = Page.objects.get(supertag=supertag, org=get_org())
        except (Page.DoesNotExist):
            msg = 'Page "%s" does not exist'
            logger.warning(msg, supertag)
            return HttpResponseNotFound(msg % supertag)

        try:
            page = page.redirect_target()
        except RedirectLoopException:
            pass

        try:
            file = File.active.get(url=url, page=page)
        except (File.DoesNotExist):
            msg = 'File "%s" on page "%s" does not exist'
            logger.warning(msg, url, page.supertag)
            return HttpResponseNotFound(msg % (url, page.supertag))

        if not has_access(supertag, request.user, current_page=get_page_by_supertag(supertag)):
            return HttpResponseForbidden('User has no access to resource')

        return response_to_download_file(file, force_download)
