
import re
from itertools import groupby

from django.http import StreamingHttpResponse
from lxml import etree

from wiki.api.views.api_view import ApiView
from wiki.api_core.deprecator import deprecated_api
from wiki.pages.models import Access, Page
from wiki.pages.utils import split_wiki_name
from wiki.utils.dictcache import DictCache
from wiki.utils.peekable import Peekable


class LockedPagesView(ApiView):
    """
    Return XML with list of restricted pages and users who has access to them
    """

    token_auth = True

    ETREE_TOSTRING_OPTIONS = dict(
        pretty_print=True,
        encoding='UTF-8',
    )

    def __init__(self, *args, **opts):
        super(LockedPagesView, self).__init__(*args, **opts)
        self.cache = DictCache()

    @deprecated_api
    def get(self, request):
        restrictions = self.get_restrictions()
        xml = self.convert_to_xml(restrictions)
        return StreamingHttpResponse(xml, content_type='text/xml')

    def get_restrictions(self):
        """
        Traverse all restricted pages and yield pairs (page, allowed_logins)
        """
        supertags = self.get_supertags()
        for supertag in supertags:
            for page, subpages in self.generate_access_trees(supertags, supertag):
                allowed_logins = Access.objects.allowed_logins(page.supertag, cache=self.cache)
                if len(allowed_logins) == 0:
                    continue
                yield page, allowed_logins
                for subpage in subpages:
                    yield subpage, allowed_logins

    def get_supertags(self):
        supertags = Access.objects.supertags()
        supertags = [s for s, _ in groupby(supertags)]
        return Peekable(supertags)

    def generate_access_trees(self, supertags, supertag, pages=None):
        pages = pages or self.get_page_with_descendants(supertag)
        page = next(pages)
        subpages = []
        for next_supertag in supertags:
            subpagesx = self.take_subpages(pages, supertag, next_supertag)
            subpages.extend(subpagesx)
            if pages.has_next() and pages.peek().supertag == next_supertag:
                for tree in self.generate_access_trees(supertags, next_supertag, pages):
                    yield tree
            else:
                supertags.undo(next_supertag)
                break
        else:
            subpagesx = self.take_subpages(pages, supertag)
            subpages.extend(subpagesx)
        yield page, subpages

    def take_subpages(self, pages, supertag, next_supertag=None):
        def page_belongs_to_supertag(page):
            return page.supertag.startswith(supertag + '/') and page.supertag != next_supertag

        return pages.takewhile(page_belongs_to_supertag)

    def get_page_with_descendants(self, supertag):
        pages = Page.active.get_descendants(supertag, True).iterator()  # only?
        pages = Peekable(pages)
        return pages

    def convert_to_xml(self, restrictions):
        yield '<?xml version="1.0" encoding="UTF-8"?>\n<pages>\n'
        for page, allowed_logins in restrictions:
            yield self.generate_page_subtree(page, allowed_logins)
        yield '\n</pages>'

    def generate_page_subtree(self, page, allowed_logins):
        page_element = etree.Element('page')
        self.append_page_subelements(page_element, page, allowed_logins)
        return etree.tostring(page_element, **self.ETREE_TOSTRING_OPTIONS)

    def append_page_subelements(self, page_element, page, allowed_logins):
        subelements = ['title', 'keywords', 'url', 'tag', 'created_at', 'modified_at', 'allowed_logins']
        for subelement in subelements:
            appender_method = getattr(self, 'append_' + subelement)
            appender_method(page_element, page, allowed_logins)

    def append_title(self, page_element, page, allowed_logins):
        etree.SubElement(page_element, 'title').text = page.title

    def append_keywords(self, page_element, page, allowed_logins):
        keywords_element = etree.SubElement(page_element, 'keywords')
        if page.keywords:
            for keyword in re.findall(r'\w+', page.keywords):
                etree.SubElement(keywords_element, 'keyword').text = keyword

    def append_url(self, page_element, page, allowed_logins):
        page_absolute_uri = self.request.build_absolute_uri('/' + page.supertag)
        etree.SubElement(page_element, 'url').text = page_absolute_uri

    def append_tag(self, page_element, page, allowed_logins):
        etree.SubElement(page_element, 'tag').text = split_wiki_name(page.tag)

    def append_created_at(self, page_element, page, allowed_logins):
        etree.SubElement(page_element, 'created_at').text = str(page.created_at)

    def append_modified_at(self, page_element, page, allowed_logins):
        etree.SubElement(page_element, 'modified_at').text = str(page.modified_at)

    def append_allowed_logins(self, page_element, page, allowed_logins):
        allowed_logins_element = etree.SubElement(page_element, 'allowed_logins')
        allowed_logins_element.text = ' '.join(allowed_logins)
