import logging
from datetime import datetime

from django.http import Http404, HttpResponseBadRequest, HttpResponseForbidden
from django.utils.translation import ugettext
from django.views.generic.base import TemplateResponseMixin
from django.views.generic.list import MultipleObjectMixin

from wiki.api.views.api_view import ApiView
from wiki.api_core.deprecator import deprecated_api
from wiki.pages.models import Page

logger = logging.getLogger(__name__)


class PageIndexView(TemplateResponseMixin, MultipleObjectMixin, ApiView):
    """
    Show paginated list of wiki pages to the yandex serach robot.

    Understands 'lang' parameter as in ## lang=ru-en ##.
    """

    # Config TemplateResponseMixin
    template_name = 'page_index.html'

    # Config MultipleObjectMixin
    queryset = Page.objects.all()
    paginate_by = 200
    context_object_name = 'pages'

    # Config ApiView
    token_auth = True

    @deprecated_api
    def get(self, request):
        """
        GET request optional params:
        @lang: return translated pages ( example: lang="ru-en" )
        @from_timestamp: return pages, modified after timestamp. Timestamp in unix time.
         ( example: from_timestamp=1379931231231231234 )
        """

        if not self.request.from_yandex_server:
            logger.error(
                '403: The request is not from Yandex.server (IP: %s, User-Agent: "%s"',
                request.META.get('HTTP_X_REAL_IP', request.META['REMOTE_ADDR']),
                request.META.get('HTTP_USER_AGENT', ''),
            )
            return HttpResponseForbidden('The request is not from Yandex.server')

        self.object_list = self.get_queryset()
        allow_empty = self.get_allow_empty()

        if not allow_empty and len(self.object_list) == 0:
            # Translators: Список страниц пуст и отображение пустых страниц запрещено
            raise Http404(
                ugettext("Empty list and '{class_name}.allow_empty' is False.").format(
                    class_name=self.__class__.__name__
                )
            )

        # optional page__modified_at_for_index filter
        from_date_str = self.request.GET.get('from_timestamp')
        if from_date_str:
            try:
                from_date = datetime.fromtimestamp(int(from_date_str))
            except ValueError:
                logger.warning('wrong timestamp %s in request: %s', from_date_str, request)
                return HttpResponseBadRequest(content='wrong timestamp {0} in query'.format(from_date_str))
            else:
                self.object_list = self.object_list.filter(modified_at_for_index__gte=from_date)

        context = self.get_context_data(
            object_list=self.object_list,
            token=self.request.GET['token'],
            lang=self.request.GET.get('lang'),
            from_timestamp=from_date_str,
        )

        if self.request.GET.get('lang'):
            context['translation_handler'] = '.{lang}'.format(lang=context['lang'])
        else:
            context['translation_handler'] = None

        return self.render_to_response(context)
