"""
Ручка отвечает пользователю 302 и в Location отправляет на страницу
для создания новой заметки.

@author: chapson
"""

from urllib.parse import quote

from django.http import HttpResponse, HttpResponseRedirect
from django.utils.translation import ugettext

from wiki.api.views.api_view import ApiView
from wiki.api_core.deprecator import deprecated_api
from wiki.pages.templatetags.evil_tags import human_datetime
from wiki.pages.views import page_not_allowed
from wiki.personalisation.quick_notes import supertag_of_user_notes
from wiki.personalisation.user_cluster import NoPersonalCluster, personal_cluster
from wiki.utils import timezone


def url_of_new_note(user_cluster):
    """Вернуть URL новой заметки относительно корня Вики

    Page -> string|None

    """
    now = timezone.now()
    return '/{0}/note-{1}/.edit?add=1&title={2}'.format(
        supertag_of_user_notes(user_cluster),
        quote(timezone.make_naive_current(now).strftime('%Y-%m-%dT%H:%M:%S')),
        quote(human_datetime(now).encode('UTF-8')),
    )


class QuickNote(ApiView):
    @deprecated_api
    def get(self, request):
        """Вернуть пользователю редирект на новую заметку или 409 ответ.

        Если пользовательского кластера нет, возвращает ответ с кодом 409,
        в теле объясняет, почему заметка не может быть создана.

        """
        if not request.user.is_authenticated:
            return page_not_allowed(request)
        try:
            cluster = personal_cluster(request.user)
        except NoPersonalCluster:
            cluster = None

        if not cluster:
            # Translators: для создания заметок создайте свой пользовательский кластер
            return HttpResponse(
                status=409, content_type='text/plain', content=ugettext('quick_note:No personal cluster')
            )
        return HttpResponseRedirect(url_of_new_note(cluster))
