import logging

from django.core.cache import caches
from django.http import HttpResponse, HttpResponseRedirect
from django.utils.decorators import method_decorator
from django.views.decorators.csrf import csrf_protect

from wiki.api.views.api_view import ApiView
from wiki.api_core.deprecator import deprecated_api
from wiki.middleware.read_only import service_is_readonly
from wiki.pages.access import is_tech_admin

logger = logging.getLogger(__name__)
cache = caches['wiki_state']


def _toggle_read_only():
    """
    Toggle Wiki read-only state.
    """
    # мы используем TwoLevelCache, у которого есть параметр reliable_answer
    current_state = service_is_readonly(True)

    assert isinstance(current_state, bool), 'Type of read-only state must be bool'

    toggled = not current_state
    _switch_read_only(state=toggled)


def _switch_read_only(state=False):
    """
    Switch wiki to read_only or return to normal

    @type state: bool
    """
    cache.set('read_only', bool(state), timeout=None)

    logger.warning('Wiki is now in {state} state'.format(state=state and 'read-only' or 'normal'))


class ToggleReadOnly(ApiView):
    """
    Toggle Wiki read-only state.
    Only admins are allowed to use this handle
    """

    http_method_names = ('post',)

    @deprecated_api
    @method_decorator(csrf_protect)
    def post(self, request, *args, **kwargs):
        if not is_tech_admin(request.user):
            logger.warn('User is not an admin "%s"', request.user.username)
            return HttpResponse(status=403)

        _toggle_read_only()

        return HttpResponseRedirect(request.META.get('HTTP_REFERER', '/'))
