from django.utils.translation import ugettext_lazy as _

from wiki.api_core.errors.rest_api_error import RestApiError
from wiki.utils.timezone import make_naive_current


class UserHasNoAccess(RestApiError):
    """
    У пользователя нет прав выполнить данное действие.
    """

    status_code = RestApiError.STATUSES.HTTP_403_FORBIDDEN
    error_code = 'USER_HAS_NO_ACCESS'
    _non_field_messages = [
        # Translators:
        #  ru: У вас нет доступа к запрошенному ресурсу
        #  en: You have no access to requested resource
        _('You have no access to requested resource')
    ]
    debug_message = 'user has no access to resource'


class FeatureDisabled(RestApiError):
    error_code = 'FEATURE_DISABLED'
    debug_message = 'Feature disabled'


class AlreadyRequestedAccess(UserHasNoAccess):
    """
    У пользователя нет прав на просмотр страницы. Содержит информацию о предыдущем запросе доступа.
    """

    error_code = 'NO_ACCESS_BUT_ALREADY_REQUESTED'

    def __init__(self, access_requested_at, access_request_reason):
        if access_requested_at:
            # Translators:
            #  ru: Вы уже запросили доступ {date} и написали "{reason}"
            #  en: You have requested access at {date} and you wrote "{reason}"
            msg = _('You have requested access at {date} and you wrote "{reason}"')

            msg = msg.format(
                date=make_naive_current(access_requested_at).strftime('%Y-%m-%d %H:%M:%S'),
                reason=access_request_reason.replace('\n', ' '),
            )

        super(AlreadyRequestedAccess, self).__init__(msg)


class AlreadyHasAccessError(RestApiError):
    """
    У пользователя уже есть доступ к запрашиваемому ресурсу.

    Запрашивать доступ повторно не нужно.
    """

    error_code = 'ALREADY_HAS_ACCESS'
    debug_message = 'The user already has access to the resource'
    _non_field_messages = [
        # Translators:
        #  ru: У вас уже есть доступ к этой странице
        #  en: You already do have access to resource
        _('You already do have access to resource')
    ]


class ExternalAccessDenied(UserHasNoAccess):
    """
    Действие недоступно вне интранета.
    """

    error_code = 'NO_EXTERNAL_ACCESS'
    debug_message = 'Not available outside intranet'


class UserDoesNotHaveWiki(UserHasNoAccess):
    """
    Сервис Wiki недоступен для пользователя.
    """

    def __init__(self, reason=None, *args, **kwargs):
        if reason:
            self.reason = reason

        super(UserDoesNotHaveWiki, self).__init__(*args, **kwargs)

    status_code = RestApiError.STATUSES.HTTP_404_NOT_FOUND
    error_code = 'USER_DOES_NOT_HAVE_WIKI'
    reason = 'external-user'
    debug_message = 'you should probably sign in with another account'


class OrgOutOfSync(UserHasNoAccess):
    """
    Организация пользователя не найдена в базе Wiki.
    Нужен вызов форсированного синка данных организации.
    """

    status_code = RestApiError.STATUSES.HTTP_403_FORBIDDEN
    error_code = 'FORCED_SYNC_REQUIRED'
    debug_message = 'Organization is out of sync'


class DontCareAboutThisOrg(RestApiError):
    """
    Организация пользователя не найдена в базе Wiki.
    Но так как это пинг синка из директория, мы говорим 200
    """

    status_code = RestApiError.STATUSES.HTTP_200_OK
    error_code = 'DONT_CARE_ABOUT_UPDATE'
    debug_message = 'Organization is unknown; _api/dir/changed is ignored'


class UserOutOfSync(UserHasNoAccess):
    """
    Пользователь не найден в базе Wiki.
    Нужен вызов форсированного синка данных пользователя.
    """

    status_code = RestApiError.STATUSES.HTTP_403_FORBIDDEN
    error_code = 'FORCED_SYNC_REQUIRED'
    debug_message = 'User is out of sync'
