import json
import logging

from cached_property import cached_property
from django.utils.translation import ugettext_lazy as _
from rest_framework import status
from rest_framework.exceptions import APIException

from wiki.api_core.logic.errors import pop_non_field_rf_errors
from wiki.users.core import get_support_contact
from wiki.utils.logic import strings_to_list

from django.http import HttpResponse

logger = logging.getLogger(__name__)


class RestApiError(APIException):
    """
    Базовая ошибка.

    У наследников должен быть определен уникальный error_code.

    Не плодите уникальные error_code, нужно стараться использовать существующие.
    Потому что использовать бекэнд удобнее, когда error_code мало.
    Часто хватает ошибки InvalidDataSentError.
    """

    STATUSES = status

    error_code = 'GENERIC_ERROR'
    status_code = STATUSES.HTTP_409_CONFLICT

    # Сообщение об ошибке для разработчика, не для пользователя.
    # Указывать обязательно.
    # Показывать клиенту не нужно.
    debug_message = ''

    # Словарь вида "имя поля из запроса" - "список локализованных строк о том, как исправить ошибку"
    _field_errors = None

    # Список локализованных сообщений об ошибке, которые не относятся ни к одному полю из запроса.
    # Пример: сервис находится в рид-онли.
    _non_field_messages = None

    # Строка, дополняющая error_code. Определяет причину возникновения ошибочной ситуации.
    reason = None

    # Детали - наполняется через конструктор
    details = None

    def __init__(self, errors_or_message=None, non_field_messages=None, debug_message=None, details=None):
        """
        @param debug_message: строка на английском языке, сообщение об ошибке, которое нужно для отладки.

        @param errors_or_message: словарь с информацией о деталях ошибки: ключ – строка, индентификатор детали,
                       значение - строка, пояснение, в чём проблема с этой деталью. Или строка,
                       тогда она считается non_field_messages, и non_field_messages нужно не указывать.
        @type errors_or_message: dict|str

        @type non_field_messages: str|list
        @param non_field_messages: Сообщение в виде строки или список сообщений об ошибках.

        Следующие вызовы одинаковы:
        raise RestApiError('error')
        raise RestApiError(non_field_messages='error')

        Т.е. если у вас нет errors, но есть non_field_messages, то достаточно указать non_field_messages
        первым аргументом.

        Про rest_framework:
        Если в rest_framework бросить ValidationError в методе validate сериализатора, то она
        не будет относиться ни к одному из полей. Все ошибки, которые возникли в сериализаторе
        и не относятся к конкретным полям, попадут из errors_or_message в non_field_messages.
        """
        extra_non_field_messages = []
        if errors_or_message is not None:
            if isinstance(errors_or_message, dict):
                # для удобства написания кода, при написании легко забыть
                # что вместо InvalidDataSentError({'field': 'error'})
                # надо писать InvalidDataSentError({'field': ['error']})
                # они даже при чтении кода слабо различимы.
                for key, value in errors_or_message.items():
                    if hasattr(value, 'capitalize'):
                        errors_or_message[key] = strings_to_list(value)
                extra_non_field_messages.extend(pop_non_field_rf_errors(errors_or_message))
                self._field_errors = errors_or_message
            elif hasattr(errors_or_message, 'capitalize'):  # what possibly it is if not a string?
                if non_field_messages:
                    # message для пользователя можно передавать
                    # либо в errors_or_message, либо в non_field_messages,
                    # но не сразу в обоих.
                    raise ValueError(
                        'You are trying to instantiate {0} with two conflicting messages "{1}" and "{2}"'.format(
                            self.__class__.__name__, errors_or_message, non_field_messages
                        )
                    )
                non_field_messages = errors_or_message
            else:
                raise ValueError('errors should be a dict or a string, you gave me {0}'.format(type(errors_or_message)))

        if non_field_messages is not None:
            extra_non_field_messages.extend(strings_to_list(non_field_messages))

        if extra_non_field_messages:
            try:
                self._non_field_messages = extra_non_field_messages
            except AttributeError:
                # Для UnhandledException невозможно переопределить _non_field_messages атрибут
                debug_message = debug_message or extra_non_field_messages

        self.debug_message = debug_message or self.debug_message
        if not self.debug_message:
            logger.warning('Debug message is empty, please specify. Class "%s"', self.__class__.__name__)

        self.details = details

        assert hasattr(self, 'error_code'), 'Class %s must have "error_code" attribute' % type(self)

    def __repr__(self):
        field_errors = ''
        if self.errors:
            field_errors = '; '.join(field + ': ' + ', '.join(errors) for field, errors in self.errors.items())
        non_field_errors = ''
        if self.non_field_messages:
            non_field_errors = ','.join(self.non_field_messages)
        return 'field errors: "{field}", non-field_errors: "{nonfield}", debug message: "{debug}"'.format(
            field=field_errors,
            nonfield=non_field_errors,
            debug=self.debug_message,
        )

    def lazy_strings_to_unicode(self, item):
        """
        Привести все ленивые объекты к unicode

        Ленивые джанго-строки из ugettext на печати выглядят как
        django.utils.functional.__proxy__, это неудобно для repr.
        """
        if item is None:
            return item
        if isinstance(item, dict):
            for key in item:
                item[key] = self.lazy_strings_to_unicode(item[key])
            return item
        elif not isinstance(item, str) and hasattr(item, '__iter__'):
            return [value.replace('\n', '\\n') for value in map(str, item)]
        else:
            return str(item).replace('\n', '\\n')

    @cached_property
    def errors(self):
        return self.lazy_strings_to_unicode(self._field_errors)

    @cached_property
    def non_field_messages(self):
        return self.lazy_strings_to_unicode(self._non_field_messages)

    def __str__(self):
        return self.__repr__()

    def as_dict(self):
        data = {
            'debug_message': self.debug_message,
            'error_code': self.error_code,
            'level': 'ERROR',
        }
        if self.errors is not None:
            data['errors'] = self.errors
        if self.non_field_messages:
            data['message'] = self.non_field_messages
        if self.reason:
            data['reason'] = self.reason
        if self.details:
            data['details'] = self.details
        return data

    def as_http_response(self):
        return HttpResponse(
            content=json.dumps(self.as_dict()),
            content_type='application/json',
            status=self.status_code,
        )

    def as_apiv2_dict(self):
        data = {'error_code': self.error_code, 'debug_message': self.debug_message, 'details': self.details}

        return data


class UnhandledException(RestApiError):
    """
    Неизвестная ошибка.
    Попытка создания нового экземпляра этого класса с переопределением сообщения об ошибке приведет к исключительной
    ситуации AttributeError("can't set attribute",), так как атрибут _non_field_messages в этом классе является
    свойством, определяющим свое значение в runtime.
    """

    status_code = RestApiError.STATUSES.HTTP_500_INTERNAL_SERVER_ERROR
    error_code = 'UNKNOWN_ERROR'
    debug_message = 'Unhandled exception'

    @property
    def _non_field_messages(self):
        return [
            # Translators:
            #  ru: Неизвестная ошибка, обратитесь за помощью к %(email_contact)s, пожалуйста
            #  en: Unknown error, contact %(email_contact)s for help, please
            _('Unknown error, contact %(email_contact)s for help, please')
            % {'email_contact': get_support_contact()}
        ]


class ResourceIsMissing(RestApiError):
    """
    Отсутствует ресурс или его необходимый компонент.
    """

    status_code = RestApiError.STATUSES.HTTP_404_NOT_FOUND
    error_code = 'NOT_FOUND'
    debug_message = 'Resource or it\'s component is missing'
    _non_field_messages = [
        # Translators:
        #  ru: Запрошенной вами сущности не существует
        #  en: The entity you requested does not exist
        _('This resource does not exist')
    ]


class BadRequest(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_400_BAD_REQUEST
    error_code = 'BAD_REQUEST'
    debug_message = 'Bad request'


class UnauthorizedError(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_401_UNAUTHORIZED
    error_code = 'UNAUTHORIZED'
    debug_message = 'Unauthorized'


class ForbiddenError(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_403_FORBIDDEN
    error_code = 'FORBIDDEN'
    debug_message = 'Unauthorized'


class ResourceAlreadyExists(RestApiError):
    """
    Ресурс уже существует.
    """

    error_code = 'ALREADY_EXISTS'
    debug_message = 'Resource already exists'
    # Translators:
    #  ru: Этот ресурс уже существует
    #  en: This resource already exists
    _non_field_messages = [_('This resource already exists')]


class WikiFormatterError(RestApiError):
    """
    Ошибка при выполнении операции в вики форматере.
    """

    status_code = RestApiError.STATUSES.HTTP_500_INTERNAL_SERVER_ERROR
    error_code = 'FORMATTER_ERROR'
    debug_message = 'Wiki Formatter returned error.'


class BillingLimitExceeded(RestApiError):
    """
    Превышен лимит режима организации.
    """

    status_code = RestApiError.STATUSES.HTTP_402_PAYMENT_REQUIRED
    error_code = 'BILLING_LIMIT_EXCEEDED'
    debug_message = 'Billing limit exceeded.'


class EndpointIsDisabled(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_409_CONFLICT
    error_code = 'ENDPOINT_IS_DISABLED'
    debug_message = 'This endpoint is unavailable in current configuration'


class MoveClusterError(RestApiError):
    """
    Ошибка при выполнении операции перемещения кластера.
    """

    error_code = 'MOVE_CLUSTER_ERROR'
    debug_message = 'Some error was occurred during the cluster move'


class RequestProcessing(RestApiError):
    status_code = 102
    error_code = 'REQUEST_PROCESSING'
    debug_message = 'The server has accepted the request, but has not completed it yet'
