import logging
import time

import waffle
from django.contrib.auth.models import AnonymousUser
from django.core.signals import request_finished
from django.http import (
    Http404,
    HttpResponse,
    HttpResponseForbidden,
    HttpResponseNotFound,
    HttpResponseRedirect,
    StreamingHttpResponse,
)
from rest_framework import fields
from rest_framework.exceptions import (
    APIException,
    MethodNotAllowed,
    NotAcceptable,
    NotFound,
    ParseError,
    ValidationError,
)
from rest_framework.generics import GenericAPIView
from rest_framework.response import Response
from ylog.context import log_context
from wiki.api_core.authentication import SimplifiedSessionAuthentication
from wiki.api_core.errors.bad_request import ContentTypeMismatch, InvalidDataSentError
from wiki.api_core.errors.permissions import FeatureDisabled, UserHasNoAccess
from wiki.api_core.errors.rest_api_error import ResourceIsMissing, RestApiError, UnhandledException
from wiki.api_core.objects import DebugRequest
from wiki.api_core.raises import raises
from wiki.api_core.read_only_utils import raise_503_on_readonly
from wiki.api_core.serializers import (
    DebugResponseSerializer,
    ErrorResponseSerializer,
    NoPageViewSerializer,
    PageRedirectSerializer,
)
from wiki.api_core.utils import is_tvm_authentication
from wiki.api_frontend.serializers.io import StatusSerializer
from wiki.org import org_ctx
from wiki.pages.access import has_access, is_admin
from wiki.utils.injector import clear
from wiki.utils.request_logging.context import extract_normalized_route

SKIP_DOCUMENTATION = 'skip_documenation'
logger = logging.getLogger(__name__)


class Redirect(Exception):
    """
    Редирект на другой тэг.
    Не наследуется от RestAPIError, потому что формально редирект не является ошибкой,
    и должен вернуть обычный 200 ответ.
    """

    def __init__(self, redirect_to):
        self.redirect_to = redirect_to


def rest_api_error_to_dict(error):
    """
    @type error: errors.RestApiError
    @rtype: dict
    """
    return error.as_dict()


class WikiAPIView(GenericAPIView):
    """
    Базовый класс для всех API View

    Из этих вьюх надо с осторожностью возвращать наследников HttpResponse.
    Не все из django.http.* будут обработаны корректно.

    Корректно будут обработаны указанные в PLAIN_RESPONSE_CLASSES.
    """

    FEATURE_FLAG = None

    authentication_classes = (SimplifiedSessionAuthentication,)

    PLAIN_RESPONSE_CLASSES = (HttpResponseRedirect, HttpResponse, HttpResponseNotFound, HttpResponseForbidden)

    # для удобства, чтобы не нужно было импортировать.
    non_200_responses = raises

    # "возвращать 503 при рид-онли режиме" в данных методах
    # можно переопределить в наследнике
    methods_protected_from_readonly = {'post', 'delete', 'put'}

    available_content_types = ('application/json', 'application/x-www-form-urlencoded')
    check_content_type_for = ('POST', 'PUT')
    render_blank_form_for_methods = ()

    no_such_page_serializer_class = NoPageViewSerializer

    @raises()
    def options(self, request, *args, **kwargs):
        return self.http_method_not_allowed(request, *args, **kwargs)

    setattr(options, SKIP_DOCUMENTATION, True)

    def __init__(self, *args, **kwargs):
        super(WikiAPIView, self).__init__()

        for method_name in (m.lower() for m in self.allowed_methods):
            try:
                method = getattr(self, method_name)
            except AttributeError:
                continue

            if method_name in self.methods_protected_from_readonly:
                setattr(self, method_name, raise_503_on_readonly(method))

    def initial(self, request, *args, **kwargs):
        if self.FEATURE_FLAG:
            if not waffle.switch_is_active(self.FEATURE_FLAG):
                raise FeatureDisabled()

        logger.debug('WikiAPIView.initial in')

        if is_tvm_authentication(request):
            if not request.yauser.is_authenticated():
                # при аутентификации по tvm тикетам объект user может быть не определен, если к нам пришел сервис
                raise UserHasNoAccess(non_field_messages='Service is not authenticated')
        elif not request.user.is_authenticated:
            raise UserHasNoAccess(non_field_messages='User is not authenticated')

        # Сохраняем имя сработавшей вьюхи для дебага
        request.view_name = '%s.%s' % (self.__class__.__module__, self.__class__.__name__)

        self.check_content_type(request)

        super(WikiAPIView, self).initial(request, *args, **kwargs)

        logger.debug('Got request from frontend "%s"', repr(request.data))

    def dispatch(self, request, *args, **kwargs):
        route = extract_normalized_route(request)
        endpoint = self.__class__.__name__

        is_anonymous = isinstance(request.user, AnonymousUser)
        with org_ctx(request.org, raise_on_empty=not is_anonymous), log_context(route=route, endpoint=endpoint):
            return super(WikiAPIView, self).dispatch(request, *args, **kwargs)

    def check_content_type(self, request):
        """
        POST и PUT запросы только в application/json
        @raise: ParseError
        """
        if request.method in self.check_content_type_for and not any(
            [(avail_ctype in request.content_type) for avail_ctype in self.available_content_types]
        ):
            raise ContentTypeMismatch('only ' + repr(self.available_content_types))

    def handle_exception(self, exc):
        """
        Обработать исключения из хэндлера запроса
        @param exc: объект исключения
        """
        logger.debug('WikiAPIView.handle_exception in')

        # хочется обрабатывать drf ValidationError порожденные из serializer.is_valid(raise_exception=True)
        # вместо if-аков if serializer.is_valid():

        if isinstance(exc, ValidationError):
            exc = InvalidDataSentError(exc.detail)

        if isinstance(exc, (Http404, NotFound)):
            if str(exc):
                message = str(exc)
            else:
                message = ResourceIsMissing().non_field_messages
            return Response(
                {
                    'debug_message': repr(exc),
                    'error_code': ResourceIsMissing.error_code,
                    'level': 'ERROR',
                    'message': message,
                },
                status=404,
                exception=True,
            )

        if isinstance(exc, Redirect):
            return Response(
                PageRedirectSerializer({'redirect_to_tag': exc.redirect_to}).data,
                status=200,
            )

        if isinstance(exc, RestApiError) and type(exc) is not RestApiError:
            # Штатная ошибка,
            # logger.info('API error occured %s %s %s', exc.status_code, exc.__class__.__name__, exc)
            data = rest_api_error_to_dict(exc)

            logger.debug('Responded to frontend with "%s"', data)
            return Response(data, status=exc.status_code, exception=True)

        elif isinstance(exc, MethodNotAllowed):
            # logger.info('API error occured: %s', exc.detail)
            data = {
                'error_code': 'METHOD_NOT_ALLOWED',
                'level': 'ERROR',
                'debug_message': exc.detail,
                'message': UnhandledException().non_field_messages,
            }
            logger.debug('Responded to frontend with "%s"', repr(data))
            return Response(data, status=exc.status_code)

        elif isinstance(exc, APIException):
            # проблема с разбором запроса
            if isinstance(exc, ParseError):
                logger.warning('Error while parsing the request: "%s"', repr(exc))
                return Response(
                    {
                        'debug_message': exc.default_detail,
                        'error_code': 'PARSE_ERROR',
                        'level': 'ERROR',
                        'message': UnhandledException().non_field_messages,
                    },
                    status=exc.status_code,
                    exception=True,
                )
            elif isinstance(exc, NotAcceptable):
                logger.warning(
                    'Error while handling the request: "%s", available_renderers: "%s", accepted_media_type: "%s"',
                    (
                        repr(exc),
                        exc.available_renderers,
                        self.request.accepted_media_type if hasattr(self.request, 'accepted_media_type') else '',
                    ),
                )
                return Response(
                    {
                        'debug_message': exc.detail,
                        'error_code': 'NOT_ACCEPTABLE',
                        'level': 'ERROR',
                        'message': UnhandledException().non_field_messages,
                    },
                    status=exc.status_code,
                    exception=True,
                )

            # что-то просочилось из недр rest_framework
            logger.error(
                'Please, use custom errors, subclasses of "%s" instead of "%s"', repr(RestApiError), repr(exc.__class__)
            )
            data = {
                # используем код ошибки по-умолчанию
                'error_code': UnhandledException.error_code,
                'level': 'ERROR',
                'message': UnhandledException().non_field_messages,
                'debug_message': str(exc),
            }
            logger.debug('Responded to frontend with "%s"', repr(data))
            return Response(data, status=UnhandledException.status_code, exception=True)
        else:
            logger.exception('Error happened while handling request "%s"', repr(exc))
            # 500я ошибка
            data = {
                'error_code': UnhandledException.error_code,
                'level': 'ERROR',
                'debug_message': '{0}: {1}'.format(UnhandledException.debug_message, str(exc)),
                'message': UnhandledException().non_field_messages,
            }
            logger.debug('Responded to frontend with "%s"', repr(data))
            return Response(data, status=500, exception=True)

    def finalize_response(self, request, response, *args, **kwargs):
        """
        Перепаковать ответ, добавив отладочную информацию

        @param request: объект запроса к API
        @param response: объект ответа API

        @rtype: Response
        @return: перепакованный ответ
        """
        logger.debug('WikiAPIView.finalize_response in')

        if 'X-Accel-Redirect' in response:
            # Некоторые вьюхи могут возвращать настоящий или внутренний nginx редиректы,
            # тогда фронтенд превратит это в редирект на клиенте.
            return response

        if isinstance(response, StreamingHttpResponse):
            return super(WikiAPIView, self).finalize_response(request, response, *args, **kwargs)

        if any(type(response) is candidate for candidate in self.PLAIN_RESPONSE_CLASSES):
            # Также некоторые вьюхи могут возвращать кастомный HttpResponse
            # (например, экспорт в файл), тогда тоже нужно вернуть его как есть.
            if response.get('Content-Type'):
                return response

        debug_response_data = {'user': request.user}

        if 199 < response.status_code < 300:
            debug_response_data.update({'data': response.data})

            serializer_cls = DebugResponseSerializer
        elif isinstance(response, Response):
            # у нас честный не-200 ответ от rest-framework API
            debug_response_data['error'] = response.data
            serializer_cls = ErrorResponseSerializer

        else:
            # у нас не 200 ответ от django-вьюхи
            # TODO: Выпилить это место, когда уберем экшены без честных Response-ответов.
            # ответ с ошибкой может прийти как из REST API, так и от старых хэндлеров
            response_data = getattr(response, 'detail', None)
            if not response_data:
                response_data = getattr(response, 'data', {'detail': None})['detail']

            if not response_data:
                response_data = getattr(response, 'content', None)

            if not response_data:
                logger.warning('Error response message is empty', exc_info=True)

            debug_response_data['error'] = {'status_code': response.status_code, 'message': response_data}

            serializer_cls = ErrorResponseSerializer

        # считает время выполнения запроса
        execution_time = int((time.time() - request.start_time) * 1000)
        debug_request_data = DebugRequest(request, execution_time, getattr(self, 'page', None))
        debug_response_data['debug'] = debug_request_data

        new_response = Response(data=serializer_cls(debug_response_data).data, status=response.status_code)
        for cookie in response.cookies.values():
            new_response.set_cookie(cookie.key, cookie.value)

        return super(WikiAPIView, self).finalize_response(request, new_response, *args, **kwargs)

    @staticmethod
    def _get_bool(request, param_name, default_value=None):
        """
        Привести строку из POST запроса к boolean виду.
        В запросах с Content-Type = 'application/x-www-form-urlencoded'
        'true' и 'false' значения не приводятся по умолчанию к boolean типу.

        @param request: объект запроса
        @param param_name: имя приводимого параметра
        @rtype: boolean
        @raise: ParseError
        """
        value = request.data.get(param_name)

        if value is None:
            return default_value

        if isinstance(value, bool):
            return value

        if value == 'true':
            return True
        elif value == 'false':
            return False
        else:
            raise InvalidDataSentError(
                'Incorrect {0} value, must be "true" or "false". Got "{1}"'.format(param_name, value)
            )

    @raises()
    def head(self, request, *args, **kwargs):
        return self.http_method_not_allowed(request, *args, **kwargs)

    setattr(head, SKIP_DOCUMENTATION, True)

    def get_serializer_class(self):
        """
        То же, что и в базовом классе, но без
        assert self.serializer_class is not None
        потому что иногда не имеет смысла писать serializer_class.
        """
        return self.serializer_class

    def validate(self, serializer_kwargs=None, context=None, data=None):
        serializer_kwargs = serializer_kwargs or {}
        serializer_class = self.get_serializer_class()
        serializer = serializer_class(data=data or self.request.data, context=context, **serializer_kwargs)

        if not serializer.is_valid():
            raise InvalidDataSentError(serializer.errors)
        return serializer.validated_data

    def build_response(self, instance=None, data=fields.empty, context=None, **kwargs):
        serializer_class = kwargs.pop('serializer_class', self.get_serializer_class())
        serializer = serializer_class(instance, data, context=context, **kwargs)
        return Response(serializer.data)

    def build_status_response(self, success, message=None, **kwargs):
        instance = {'success': success}
        if message:
            instance['message'] = message
        return self.build_response(instance, serializer_class=StatusSerializer, **kwargs)

    def build_success_status_response(self, message=None):
        return self.build_status_response(success=True, message=message)


class PageAPIView(WikiAPIView):
    """
    Базовый класс для API View, завязанных на страницы
    """

    no_such_page_serializer_class = NoPageViewSerializer
    expected_page_type = None
    available_for_admins = False
    check_readonly_mode = False

    def initial(self, request, *args, **kwargs):
        super(PageAPIView, self).initial(request, *args, **kwargs)

        # пытаемся понять, к какой странице относится запрос
        self.check_page_exists()

        # проверить права пользователя для редактирования, если у страницы включен режим только для чтения
        self.check_readonly_access()

        # и есть ли права у пользователя на эту страницу
        self.check_page_access()

        # хук для проверки доступов для отдельных методов
        self.check_method_access()

        # проверить тип страницы если задан
        self.check_expected_page_type()

    def check_page_exists(self):
        """
        Проверить наличие существующей страницы в запросе, и отреагировать соответственно
        @raise: Http404
        """
        # Если тег страницы некорректен, его может исправить
        # middleware.RecognizePageMiddleware

        if not self.request.page:
            raise Http404('Page cannot be found')

    def check_page_access(self):
        """
        Проверить возможность доступа пользователя из запроса к странице
        @raise UserHasNoAccess
        """
        if self.request.from_yandex_server:
            return
        if self.available_for_admins and is_admin(self.request.user):
            return
        if has_access(self.request.supertag, self.request.user, privilege='read', current_page=self.request.page):
            return
        raise UserHasNoAccess(non_field_messages='User is not in the ACL for requested page')

    def check_readonly_access(self):
        http_method = self.request.method.upper()
        is_rw_method = http_method in ('POST', 'PUT', 'DELETE')

        if self.request.page and self.request.page.is_readonly and self.check_readonly_mode and is_rw_method:
            self.check_owner_access()

    def check_owner_access(self):
        """
        Проверка, что пользователь админ или автор страницы
        """
        if not (self.request.user in self.request.page.get_authors() or is_admin(self.request.user)):
            raise UserHasNoAccess(non_field_messages='Access only for page authors')

    def check_method_access(self):
        method = self.request.method.lower()
        checker = getattr(self, 'check_access_' + method, lambda *a, **kw: None)
        checker()

    def check_expected_page_type(self):
        """
        Проверить правильность типа страницы, к которой делается запрос.
        @raise ContentTypeMismatch
        """
        if self.expected_page_type is None:
            return

        page = self.request.page
        if page.page_type == self.expected_page_type:
            return

        msg = 'Page "{supertag}" has type "{type}", expected "{expected}"'.format(
            supertag=page.supertag,
            type=page.page_type,
            expected=self.expected_page_type,
        )
        raise ContentTypeMismatch(msg)


# очищаем инжектор в конце запроса
request_finished.connect(clear)


class NoSuchPageView(WikiAPIView):
    """
    View возвращает 404 ответ.

    Например,

    %%
    curl -H "Authorization: OAuth <token>" -H "Content-Type: application/json" \
    "https://wiki-api.yandex-team.ru/_api/frontend/.files"
    %%

    %%(js)
    {
        "error": {
            "message": "Http404('Page cannot be found',)",
            "error_code": "NOT_FOUND"
        }
    }
    %%

    """

    serializer_class = NoPageViewSerializer

    def initial(self, request, *args, **kwargs):
        super(NoSuchPageView, self).initial(request, *args, **kwargs)
        raise Http404('This URL does not match any view, so you got default answer (404)')
