import base64
import logging
import uuid
from datetime import timedelta
from hashlib import sha256

from django.conf import settings
from django.core.cache import caches
from django.utils.translation import ugettext as _

from wiki.files.models import MDS_STORAGE, File, make_storage_id
from wiki.grids.logic import grids_import
from wiki.notifications.models import EventTypes, PageEvent
from wiki.utils import timezone
from wiki.utils.limits import limit__attach_size
from wiki.utils.supertag import translit

logger = logging.getLogger(__name__)

cache_source = caches['imported_grids']


class FileUploadError(Exception):
    def __str__(self):
        raise NotImplementedError


class NoFilesGivenError(FileUploadError):
    def __str__(self):
        # Translators:
        #  ru: В запросе не было файлов
        #  en: No files were given in request
        return _('No files were given')


class EmptyFileError(FileUploadError):
    def __str__(self):
        # Translators:
        #  ru: Файл пустой
        #  en: The file is empty
        return _('The file is empty')


class SuchBigFileError(FileUploadError):
    MAX_FILE_SIZE = settings.FILES_UPLOAD_MAX_SIZE

    def __init__(self, max_file_size=MAX_FILE_SIZE):
        self.max_file_size = max_file_size

    def __str__(self):
        # Translators:
        #  ru: Файл слишком большой, максимально допустимый размер "%s"
        #  en: The file is too big to be uploaded, max is "%s"
        return _('The file is too big to be uploaded, max is "%s"') % f'{self.max_file_size / float(1 << 20):,.0f} MB'


def _check_file_size(file_data, max_file_size):
    if file_data.size == 0:
        raise EmptyFileError()
    if file_data.size > max_file_size:
        raise SuchBigFileError(max_file_size)


def _get_first_file_from_request(request, raise_on_empty=True):
    """
    Вернуть первый файл из объекта HttpRequest.

    Вспомогательная функция. Если файлов нет, бросит исключение.

    @type request: HttpRequest
    @rtype: UploadedFile
    """
    # в запросе должен присутствовать хотя бы один файл
    if not request.FILES or not list(request.FILES.values()):
        if raise_on_empty:
            message = 'no files given in request'
            logger.warning(message)
            raise NoFilesGivenError(message)
        else:
            return None

    return list(request.FILES.values())[0]


def upload_file_to_cache(request):
    """
    Загрузить первый файл из объекта HttpRequest в джанго кэш.

    @type request: HttpRequest
    @rtype: tuple
    @return: ключ в кэше, имя файла, размер файла
    """
    file_data = _get_first_file_from_request(request)

    extension = grids_import.get_file_extention(file_data.name)
    grids_import.check_file_extention(extension)

    _check_file_size(file_data, settings.IMPORT_FILE_MAX_SIZE)

    cache_name = translit(file_data.name) + str(uuid.uuid4())
    cache_key = sha256(cache_name.encode()).hexdigest()

    cache_source.set(
        cache_key, {'name': file_data.name, 'contents': file_data if isinstance(file_data, str) else file_data.read()}
    )

    return cache_key, file_data.name, file_data.size


def upload_file_to_storage(request):
    """
    Загрузить первый файл из объекта HttpRequest в storage.

    @type request: HttpRequest
    @rtype: tuple
    @return: ключ в хранилище
    """
    file_data = _get_first_file_from_request(request)
    _check_file_size(file_data, SuchBigFileError.MAX_FILE_SIZE)
    return upload_file_data_to_storage(file_data, file_data.size, file_data.name)


def upload_file_data_to_storage(file_data, file_size, file_name):
    file_size_mb = file_size / (1 << 20)
    if limit__attach_size.exceeded(file_size_mb):
        from wiki.api_core.errors.rest_api_error import BillingLimitExceeded

        display_limit = limit__attach_size.get() / 1024.0
        if int(display_limit) == display_limit:
            display_limit = int(display_limit)
        else:
            display_limit = round(display_limit, 3)
        raise BillingLimitExceeded(
            # Translators:
            #  ru: Достигнут лимит на объем файлов: {max_attach_size_gb} Гб.
            #  Подробнее: https://connect.yandex.ru/pricing/connect
            #  en: You have reached files size limit: {max_attach_size_gb} Gb.
            #  Read more: https://connect.yandex.ru/pricing/connect
            _('billing.limits:Files size limit {max_attach_size_gb} Gb').format(max_attach_size_gb=display_limit)
        )

    storage_id = make_storage_id(file_name, file_size)
    logger.info('saving file "%s" id=%s to storage', file_name, storage_id)
    storage_key = MDS_STORAGE.save(storage_id, file_data)

    return storage_key


def add_file_to_page(page, user, file_storage_id, is_silent: bool = False):
    if not MDS_STORAGE.exists(file_storage_id):
        # Translators:
        #  ru: В хранилище нет файла с id "%s"
        #  en: No such file with id "%s" in storage
        raise NoFileInStorage(_('No such file with id "%s" in storage') % file_storage_id)

    NOW = timezone.now()

    words = file_storage_id.split(':')
    file_name, file_size = words[2], words[3]
    # декодируем имя файла из base64
    file_name = base64.b64decode(file_name).decode('utf-8')

    attached_file = File(
        page=page,
        user=user,
        name=file_name,
        url=File.get_unique_url(page, file_name),
        size=file_size,
        # поле description на клиенте в новом дизайне не предусмотрено
        description='',
        created_at=NOW,
        modified_at=NOW,
        mds_storage_id=file_storage_id,
    )

    attached_file.save()

    page.files += 1
    page.save()

    PageEvent(
        timeout=NOW + timedelta(minutes=20),
        page=page,
        author=user,
        meta={
            'filename': attached_file.name,
            'url': attached_file.url,
            'size': attached_file.size,
            'id': attached_file.id,
        },
        event_type=EventTypes.add_file,
        notify=not is_silent,
    ).save()

    return attached_file


class NoFileInStorage(Exception):
    pass
