import logging
import re
from typing import Optional

import waffle

from django.conf import settings
from django.contrib.auth import get_user_model
from django.http import HttpRequest, HttpResponse
from wiki.api_core.errors.rest_api_error import BadRequest
from ylog.context import log_context

from wiki.api_core.waffle_switches import ENABLE_COPY_ON_WRITE
from wiki.org import get_org, org_ctx
from wiki.pages.logic.etalon import get_etalon_page, create_etalon_page
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import as_base_organization
from wiki.utils.supertag import remove_trailing_dots, translit

logger = logging.getLogger(__name__)

User = get_user_model()

# api page supertag pattern

api_tag_pattern = re.compile(
    # Запрос в корень API (без хендлера) приравнивается к главной странице.
    # Запросы на хендлеры могут быть как привязанными к страницам, так и у корня API,
    # в последнем случае страница определяться не должна.
    r'^/_api/(frontend|v1/pages|svc/pages)(/+)((?P<empty_tag>(.bemjson)?$)|(?P<action>.actions/[\w]+$)|%s(/(\..+)?)?|(\..+))'  # noqa
    % settings.TAG_REGEX,
    re.UNICODE,
)


class RecognizePageMiddleware:
    """
    Распознать страницу в запросе
    """

    def __init__(self, get_response=None):
        self.get_response = get_response

    def __call__(self, request: HttpRequest) -> HttpResponse:
        response = self.process_request(request)

        with log_context(slug=getattr(request, 'supertag')):
            if response:
                return response
            return self.get_response(request)

    @staticmethod
    def process_request(request: HttpRequest) -> Optional[HttpResponse]:
        logger.debug('RecognizePageMiddleware in')

        request.tag = None
        request.supertag = None
        request.page = None

        tag = RecognizePageMiddleware.parse_tag(request)
        if not tag:
            return

        request.tag = tag
        request.supertag = translit(tag)

        proper_tag = remove_trailing_dots(tag)
        if proper_tag != tag:
            request.redirect_to = proper_tag
            return

        try:
            if settings.IS_BUSINESS and request.org is None:  # ---- иначе будет 500 NOORG
                return BadRequest(debug_message='"X-Org-Id" header value is required').as_http_response()

            with org_ctx(request.org):
                request.page = Page.objects.prefetch_related('last_author', 'authors').get(
                    supertag=request.supertag, status__gt=0, org=get_org()
                )
        except Page.DoesNotExist:
            RecognizePageMiddleware.process_copy_on_write(request)

    @staticmethod
    def process_copy_on_write(request: HttpRequest):
        # Если запрошенная страница не существует но для неё возможен copy on write
        # пробуем получить соответствующую эталонную страницу
        if (
            waffle.switch_is_active(ENABLE_COPY_ON_WRITE)
            and request.page is None
            and request.supertag in settings.COPY_ON_WRITE_TAGS
        ):
            organization = as_base_organization(request.org)

            if request.method in ['GET', 'HEAD']:
                etalon_page = get_etalon_page(organization, slug=request.supertag)
            else:
                etalon_page = create_etalon_page(organization, slug=request.supertag)

            request.page = etalon_page

    @staticmethod
    def parse_tag(request: HttpRequest) -> Optional[str]:
        tag_matching = api_tag_pattern.search(request.META['PATH_INFO'])

        if not tag_matching:
            # Какой-то непонятный запрос, нечего парсить
            return None

        if tag_matching.group('empty_tag') is not None or tag_matching.group('action'):
            # Запрос на корень API без хендлера или на корневой хендлер с каким-то экшином - показываем главную страницу
            tag = settings.MAIN_PAGE
        elif tag_matching.group('tag'):
            tag = tag_matching.group('tag')
        else:
            # Запрос на корневой хендлер, например, /_api/frontend/.autocomplete, страницу не матчим
            return None

        return tag.rstrip('/')
