
import logging
from datetime import datetime
import email.utils as eut
from functools import wraps

from django.http import HttpResponseNotModified

from wiki.utils.timezone import make_aware_utc

# явный логгер, чтобы избежать потерь логгирования при смене модуля.
logger = logging.getLogger('wiki.api_core.not_modified')


def was_modified(request, page):
    """
    Вернуть True, если надо ответить 304.

    Значение заголовка If-Modified-Since должно
    быть в формате "Fri, 08 Aug 9014 11:09:43 GMT".
    """
    if 'HTTP_IF_MODIFIED_SINCE' not in request.META:
        # думаем, что документ был изменен. Потому что не можем определить.
        return True
    client_dt = datetime(*eut.parsedate(request.META['HTTP_IF_MODIFIED_SINCE'])[:6])
    # проставляем таймзону
    client_dt = make_aware_utc(client_dt)
    logger.info('Page was modified at "%s", client has last seen this page at "%s"', page.modified_at, client_dt)
    return page.modified_at > client_dt


def send_304_if_not_modified(method):
    """
    Декоратор отправляет 304ый ответ, если страница не была модифицирована.
    """

    @wraps(method)
    def wrapper(self, request, *args, **kwargs):
        if not was_modified(request, request.page):
            return HttpResponseNotModified()
        return method(self, request, *args, **kwargs)

    return wrapper
