import logging
from functools import wraps

from wiki.api_core.errors.bad_request import ContentTypeMismatch, InvalidDataSentError
from wiki.api_core.errors.permissions import FeatureDisabled, UserHasNoAccess
from wiki.api_core.errors.rest_api_error import ResourceIsMissing, RestApiError

# флаг, который выставляет декоратор raises

ERRORS_THAT_CAN_BE_RAISED = 'API_ERRORS_THAT_CAN_BE_RAISED'
ALLOWED_BASE_CLASSES = (RestApiError,)


def can_raise_anything(fn):
    setattr(fn, ERRORS_THAT_CAN_BE_RAISED, set())
    return fn


def raises(*args):
    """
    Обязательный для API декоратор, принимающий список ошибок, которые бросает ручка.

    Этот декоратор всегда должен быть установлен последним:
    Если после него какой-то декоратор бросит наследника RestApiError,
    то raises его не поймает.
    Сделано для неудобства разработчика.

     --- На выпил ---

    https://github.yandex-team.ru/tools/wiki/pull/828#issuecomment-71464

    Если он не указан, ручка не сможет быть вызвана через API.
    """
    possible_errors = set()

    for error_class in args:
        if issubclass(error_class, ALLOWED_BASE_CLASSES):
            possible_errors.add(error_class)
        else:
            logging.error('Exception %s is not derived from allowed base classes. It will not be used' % error_class)

    possible_errors.add(ResourceIsMissing)
    possible_errors.add(ContentTypeMismatch)
    possible_errors.add(InvalidDataSentError)
    possible_errors.add(UserHasNoAccess)
    possible_errors.add(FeatureDisabled)

    def wrap(api_method):
        @wraps(api_method)
        def check_exceptions_are_registered(self, *args, **kwargs):
            return api_method(self, *args, **kwargs)

        setattr(check_exceptions_are_registered, ERRORS_THAT_CAN_BE_RAISED, possible_errors)

        return check_exceptions_are_registered

    return wrap
