import functools
import logging

from django.db import DatabaseError
from django_replicated.utils import routers
from rest_framework.response import Response

from wiki.api_core.errors.read_only import ReadonlyError
from wiki.middleware.read_only import service_is_readonly
from wiki.utils.backports.mds_compat import APIError

logger = logging.getLogger(__name__)


def response_service_readonly(exception_message):
    """
    Вернуть ответ о недоступности сервиса на запись.

    @rtype: Response
    """
    return Response(
        {
            'message': exception_message,
            'error_code': ReadonlyError.error_code,
            'level': 'ERROR',
        },
        status=ReadonlyError.status_code,
    )


def postgres_is_readonly(exception):
    # https://www.postgresql.org/docs/current/static/errcodes-appendix.html#ERRCODES-TABLE
    # 25006	read_only_sql_transaction
    return (
        isinstance(exception, DatabaseError) and getattr(getattr(exception, '__cause__', None), 'pgcode', None) == 25006
    )


def raise_503_on_readonly(func):
    """
    Декоратор, чтобы бросать ошибку о том, что сервис перешел в рид-онли.

    """

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        if service_is_readonly() and routers.state() == 'master':
            raise ReadonlyError(non_field_messages='Please try again in 10 seconds, lost connectivity with master')
        try:
            return func(*args, **kwargs)
        except APIError:
            logger.exception('Details for "MDS is not available"')
            raise ReadonlyError(non_field_messages='MDS is not available')
        except DatabaseError as exc:
            if postgres_is_readonly(exc):
                raise ReadonlyError(non_field_messages='Database is read-only')
            raise

    return wrapper
